<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Render\Element;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\webform\WebformInterface;

/**
 * Returns the Webform elements.
 *
 * @DataProducer(
 *   id = "webform_elements",
 *   name = @Translation("Webform elements"),
 *   description = @Translation("Returns the Webform elements."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Webform elements"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "webform" = @ContextDefinition("entity:webform",
 *       label = @Translation("Webform"),
 *       required = TRUE
 *     ),
 *     "parent" = @ContextDefinition("any",
 *       label = @Translation("Parent element"),
 *       required = FALSE
 *     )
 *   }
 * )
 */
class WebformElements extends DataProducerPluginBase {

  /**
   * Resolves Webform elements from a passed in Webform or parent element.
   *
   * This data producer can be used to resolve the list of elements from a
   * Webform. Nested elements can also be retrieved by passing in the parent
   * element. The Webform module has two different ways to nest elements inside
   * other elements: "composite elements" and "containers". They work in subtly
   * different ways. This takes care of both variations.
   *
   * @param \Drupal\webform\WebformInterface $webform
   *   The full webform.
   * @param array|null $parent
   *   An optional parent element. This can be either a composite element or a
   *   container. If this is passed in, the child elements will be resolved. If
   *   this is NULL, the top-level elements from the $webform argument will be
   *   resolved.
   *
   * @return array
   *   An array of Webform render array elements.
   *
   * @see \Drupal\webform\Plugin\WebformElement\ContainerBase
   * @see \Drupal\webform\Plugin\WebformElement\WebformCompositeBase
   */
  public function resolve(WebformInterface $webform, ?array $parent = NULL): array {
    // If we are passed in a parent element, retrieve the child elements from
    // it. If it is not passed we will retrieve the top-level elements from the
    // entire Webform.
    $parent_element = $parent ?? $webform->getSubmissionForm() ?? [];

    // Depending on whether this is a full webform or a composite / container
    // element, get the children from the right location.
    $original_elements = match (TRUE) {
      // If this is a composite element the children can be found in the
      // '#webform_composite_elements' property.
      // @see \Drupal\webform\Plugin\WebformElement\WebformCompositeBase::initializeCompositeElements()
      !empty($parent_element['#webform_composite_elements']) => $parent_element['#webform_composite_elements'],
      // When this is the root element of a Webform, the children can be found
      // in the 'elements' property.
      // @todo We have a potential collision here if a container element has
      //   a child identified with the 'elements' key.
      !empty($parent_element['elements']) => $parent_element['elements'],
      // If this is a container element the children can be found in the root.
      default => $parent_element,
    };

    $elements = [];
    foreach (Element::children($original_elements, TRUE) as $key) {
      $element = $original_elements[$key];
      if (!empty($element)) {
        // Some composite elements (like 'address') reuse Form API form elements
        // rather than Webform elements. Make sure to include the element ID and
        // key so that the client can identify the element.
        $element_key = $element['#webform_key'] ?? $element['#webform_composite_key'] ?? NULL;
        if (empty($element_key)) {
          $parent_key = $parent_element['#webform_key'] ?? $parent_element['#webform_composite_key'] ?? NULL;
          if ($parent_key) {
            $parent_id = $parent_element['#webform_id'] ?? $parent_element['#webform_composite_id'] ?? $webform->id();
            $element['#webform_composite_id'] = $parent_id . '--' . $key;
            $element['#webform_composite_key'] = $parent_key . '__' . $key;
          }
        }
        $elements[] = $element;
      }
    }
    return $elements;
  }

}
