<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\GraphQL\Buffers\EntityBuffer;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use GraphQL\Deferred;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Loads a webform by ID.
 *
 * @DataProducer(
 *   id = "webform_load",
 *   name = @Translation("Webform load"),
 *   description = @Translation("Load a webform by ID."),
 *   produces = @ContextDefinition("entity:webform",
 *     label = @Translation("Webform"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "id" = @ContextDefinition("string",
 *       label = @Translation("Webform ID")
 *     ),
 *     "sourceEntityType" = @ContextDefinition("string",
 *       label = @Translation("Source entity type"),
 *       required = FALSE
 *     ),
 *     "sourceEntityId" = @ContextDefinition("string",
 *       label = @Translation("Source entity ID"),
 *       required = FALSE
 *     ),
 *     "language" = @ContextDefinition("string",
 *       label = @Translation("Entity language"),
 *       required = FALSE
 *     )
 *   }
 * )
 */
class WebformLoad extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityBuffer $entityBuffer,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('graphql.buffer.entity'),
    );
  }

  /**
   * Resolves the requested webform.
   */
  public function resolve(string $id, ?string $sourceEntityType, ?string $sourceEntityId, ?string $language, ?FieldContext $context) {
    $resolver = $this->entityBuffer->add('webform', $id);

    return new Deferred(function () use ($sourceEntityType, $sourceEntityId, $language, $resolver, $context) {
      if (!$webform = $resolver()) {
        // If there is no webform with this ID, add the list cache tags so that
        // the cache entry is purged whenever a new webform is created.
        $type = $this->entityTypeManager->getDefinition('webform');
        /** @var \Drupal\Core\Entity\EntityTypeInterface $type */
        $tags = $type->getListCacheTags();
        $context->addCacheTags($tags);
        return NULL;
      }

      $context->addCacheableDependency($webform);

      // Get the correct translation.
      if (isset($language) && $language !== $webform->language()->getId() && $webform instanceof TranslatableInterface) {
        $webform = $webform->getTranslation($language);
        $webform->addCacheContexts(["static:language:{$language}"]);
      }

      if ($sourceEntityType && $sourceEntityId) {
        $context->setContextValue('webform_source_entity_type', $sourceEntityType);
        $context->setContextValue('webform_source_entity_id', $sourceEntityId);

        // Add a cacheable dependency on the source entity. If it doesn't exist,
        // add the list cache tags.
        $sourceEntity = $this->entityTypeManager->getStorage($sourceEntityType)->load($sourceEntityId);
        if ($sourceEntity) {
          $context->addCacheableDependency($sourceEntity);
        }
        else {
          $type = $this->entityTypeManager->getDefinition($sourceEntityType);
          /** @var \Drupal\Core\Entity\EntityTypeInterface $type */
          $tags = $type->getListCacheTags();
          $context->addCacheTags($tags);
        }
      }

      // Check if the current user has access to the webform.
      /** @var \Drupal\Core\Access\AccessResultInterface $accessResult */
      $accessResult = $webform->access('view', NULL, TRUE);
      $context->addCacheableDependency($accessResult);
      if (!$accessResult->isAllowed()) {
        return NULL;
      }

      return $webform;
    });
  }

}
