<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\GraphQL\Buffers\EntityBuffer;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use GraphQL\Deferred;
use GraphQL\Error\UserError;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns the webform submission.
 *
 * @DataProducer(
 *   id = "webform_load_submission",
 *   name = @Translation("Webform load submission"),
 *   description = @Translation("Returns the webform submission."),
 *   produces = @ContextDefinition("entity:webform_submission",
 *     label = @Translation("Webform submission"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "id" = @ContextDefinition("integer",
 *       label = @Translation("Webform submission ID")
 *     ),
 *     "token" = @ContextDefinition("string",
 *       label = @Translation("Webform submission token")
 *     ),
 *   }
 * )
 */
class WebformLoadSubmission extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityBuffer $entityBuffer,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $pluginId, $pluginDefinition) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('entity_type.manager'),
      $container->get('graphql.buffer.entity'),
    );
  }

  /**
   * Resolves the webform submission by ID and token.
   *
   * @param string $id
   *   The webform submission ID.
   * @param string $token
   *   The webform submission token.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $context
   *   The field context.
   *
   * @return \GraphQL\Deferred
   *   A deferred webform submission entity, or NULL if not found or token is
   *   invalid.
   */
  public function resolve(int $id, string $token, FieldContext $context): Deferred {
    $resolver = $this->entityBuffer->add('webform_submission', $id);

    return new Deferred(function () use ($token, $resolver, $context) {
      /** @var \Drupal\webform\WebformSubmissionInterface $submission */
      if (!$submission = $resolver()) {
        // If there is no webform submission with this ID. It might be added at
        // a later time, so add the list cache tags. This will ensure the cache
        // for this entry is purged whenever a new submission is created.
        $type = $this->entityTypeManager->getDefinition('webform_submission');
        /** @var \Drupal\Core\Entity\EntityTypeInterface $type */
        $tags = $type->getListCacheTags();
        $context->addCacheTags($tags);
        return NULL;
      }

      $context->addCacheableDependency($submission);

      // Check if the token is valid.
      if ($submission->getToken() !== $token) {
        throw new UserError('Invalid webform submission token.');
      }

      return $submission;
    });
  }

}
