<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\graphql_webform\Enum\ExposedSettings;
use Drupal\webform\WebformInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns the settings for a webform.
 *
 * @DataProducer(
 *   id = "webform_settings",
 *   name = @Translation("Webform settings"),
 *   description = @Translation("Returns the settings for a webform which are relevant for a decoupled frontend."),
 *   produces = @ContextDefinition("map",
 *     label = @Translation("Webform Settings"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "webform" = @ContextDefinition("entity:webform",
 *       label = @Translation("Webform")
 *     )
 *   }
 * )
 */
class WebformSettings extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $pluginId, $pluginDefinition) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('config.factory'),
    );
  }

  /**
   * Resolves the webform settings.
   *
   * @param \Drupal\webform\Entity\Webform $webform
   *   The webform to get the settings for.
   */
  public function resolve(WebformInterface $webform): array {
    $globalSettings = $this->configFactory->get('webform.settings')->get('settings') ?? [];
    $webformSettings = $webform->getSettings();
    $settings = [];

    foreach (array_merge($globalSettings, $webformSettings) as $key => $value) {
      if (str_starts_with($key, 'default_')) {
        $key = substr($key, 8);
        if ($exposedSetting = ExposedSettings::tryFrom($key)) {
          $settings[$exposedSetting->toGraphQlFieldName()] = $value;
        }
      }
    }

    return $settings;
  }

}
