<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Render\RendererInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\graphql_webform\Enum\WebformSubmissionConfirmationType;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformMessageManagerInterface;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform\WebformTokenManagerInterface;
use GraphQL\Error\UserError;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns confirmation information for a webform submission.
 *
 * @DataProducer(
 *   id = "webform_submission_confirmation",
 *   name = @Translation("Webform submission confirmation"),
 *   description = @Translation("Returns confirmation details for a webform submission."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Confirmation information"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "submission" = @ContextDefinition("entity:webform_submission",
 *       label = @Translation("Webform submission"),
 *       required = FALSE
 *     ),
 *     "webform" = @ContextDefinition("entity:webform",
 *       label = @Translation("Webform"),
 *       required = FALSE
 *     )
 *   }
 * )
 */
class WebformSubmissionConfirmation extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected readonly WebformMessageManagerInterface $messageManager,
    protected readonly WebformTokenManagerInterface $tokenManager,
    protected readonly RendererInterface $renderer,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('webform.message_manager'),
      $container->get('webform.token_manager'),
      $container->get('renderer'),
    );
  }

  /**
   * Resolves confirmation information for a webform submission.
   *
   * @param \Drupal\webform\WebformSubmissionInterface|null $submission
   *   Optional webform submission.
   * @param \Drupal\webform\WebformInterface|null $webform
   *   Optional webform entity.
   *
   * @return array
   *   The confirmation information.
   */
  public function resolve(?WebformSubmissionInterface $submission = NULL, ?WebformInterface $webform = NULL): array {
    // If neither a webform nor a submission is provided, we cannot proceed.
    if (!$webform && !$submission) {
      throw new UserError('Either a webform or a webform submission must be provided.');
    }

    $renderContext = new RenderContext();
    $renderResult = $this->renderer->executeInRenderContext($renderContext, function () use ($webform, $submission) {
      $webform = $webform ?: $submission->getWebform();
      $sourceEntity = $submission?->getSourceEntity();

      $confirmationType = WebformSubmissionConfirmationType::tryFrom($webform->getSetting('confirmation_type', '')) ?? WebformSubmissionConfirmationType::None;

      $title = (string) $webform->getSetting('confirmation_title');
      $title = $this->tokenManager->replace($title, $submission ?? $webform);

      if (is_array($title)) {
        $title = $this->renderer->renderInIsolation($title);
      }

      $this->messageManager->setSourceEntity($sourceEntity);
      $this->messageManager->setWebformSubmission($submission);
      $this->messageManager->setWebform($webform);
      $message = $this->messageManager->build(WebformMessageManagerInterface::SUBMISSION_CONFIRMATION_MESSAGE);
      $message = $this->renderer->renderInIsolation($message);

      // Pass on the confirmation redirection URL. Note that this has elaborate
      // handling in the webform module, including token replacement, stripping
      // of query arguments, resolving internal paths, etc. In decoupled setups,
      // much of this is irrelevant since we will probably not be redirecting to
      // a Drupal page, so we just pass on the raw URL here.
      $redirectUrl = $webform->getSetting('confirmation_url');

      return [
        'type' => $confirmationType->toGraphQlEnumValue(),
        'title' => !empty($title) ? (string) $title : NULL,
        'message' => (string) $message,
        'redirectUrl' => (string) $redirectUrl,
      ];
    });

    return $renderResult;
  }

}
