<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Render\RendererInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Renders the token URL for a webform submission.
 *
 * @DataProducer(
 *   id = "webform_submission_token_url",
 *   name = @Translation("Webform submission token URL"),
 *   description = @Translation("Renders the token URL fir a Webform submission."),
 *   produces = @ContextDefinition("string",
 *     label = @Translation("The rendered token URL"),
 *   ),
 *   consumes = {
 *     "submission" = @ContextDefinition("entity:webform_submission",
 *       label = @Translation("Webform submission")
 *     )
 *   }
 * )
 */
class WebformSubmissionTokenUrl extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected readonly RendererInterface $renderer,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('renderer')
    );
  }

  /**
   * Resolves the token URL for a Webform submission.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $submission
   *   The Webform submission entity.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $context
   *   The field context.
   *
   * @return string
   *   The token URL as a string.
   */
  public function resolve(WebformSubmissionInterface $submission, FieldContext $context): string {
    $renderContext = new RenderContext();

    // If submissions are not saved on the webform (setting),
    // the token URL is not available.
    if ($submission->id() === NULL) {
      return '';
    }

    $value = $this->renderer->executeInRenderContext($renderContext, function () use ($submission) {
      return $submission->getTokenUrl('view')->toString();
    });

    $context->addCacheableDependency($renderContext);

    return $value;
  }

}
