<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\webform\WebformInterface;

/**
 * Returns the title of the webform, as configured in the webform settings.
 *
 * Depending on the webform settings, this may include the webform label, the
 * label of the source entity, or both.
 *
 * @DataProducer(
 *   id = "webform_title",
 *   name = @Translation("Webform title"),
 *   description = @Translation("Returns the title of the webform."),
 *   produces = @ContextDefinition("string",
 *     label = @Translation("Webform title"),
 *     required = TRUE
 *   ),
 *   consumes = {
 *     "webform" = @ContextDefinition("entity:webform",
 *       label = @Translation("Webform"),
 *       required = TRUE
 *     ),
 *     "sourceEntity" = @ContextDefinition("entity",
 *       label = @Translation("Source entity"),
 *       required = FALSE
 *     )
 *   }
 * )
 */
class WebformTitle extends DataProducerPluginBase {

  /**
   * Resolves the title of the passed in Webform.
   *
   * @param \Drupal\webform\WebformInterface $webform
   *   The webform.
   * @param \Drupal\Core\Entity\EntityInterface|null $sourceEntity
   *   An optional source entity.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $context
   *   The field context.
   *
   * @return string
   *   The webform title.
   *
   * @see \Drupal\webform\Controller\WebformEntityController::title()
   */
  public function resolve(WebformInterface $webform, ?EntityInterface $sourceEntity, FieldContext $context): string {
    $context->addCacheableDependency($webform);

    // If source entity does not exist, always use the webform's label.
    if (!$sourceEntity) {
      return (string) $webform->label();
    }

    $context->addCacheableDependency($sourceEntity);

    // Handle duplicate titles.
    if ($sourceEntity->label() === $webform->label()) {
      return (string) $webform->label();
    }

    // Get the webform's title.
    switch ($webform->getSetting('form_title')) {
      case WebformInterface::TITLE_SOURCE_ENTITY:
        return (string) $sourceEntity->label();

      case WebformInterface::TITLE_WEBFORM:
        return (string) $webform->label();

      case WebformInterface::TITLE_WEBFORM_SOURCE_ENTITY:
        $t_args = [
          '@source_entity' => $sourceEntity->label(),
          '@webform' => $webform->label(),
        ];
        return (string) $this->t('@webform: @source_entity', $t_args);

      case WebformInterface::TITLE_SOURCE_ENTITY_WEBFORM:
      default:
        $t_args = [
          '@source_entity' => $sourceEntity->label(),
          '@webform' => $webform->label(),
        ];
        return (string) $this->t('@source_entity: @webform', $t_args);
    }
  }

}
