<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\SchemaExtension;

use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Environment;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\graphql\GraphQL\ResolverBuilder;
use Drupal\graphql\GraphQL\ResolverRegistryInterface;
use Drupal\graphql\Plugin\GraphQL\SchemaExtension\SdlSchemaExtensionPluginBase;
use Drupal\graphql_webform\Enum\WebformElementDescriptionDisplay;
use Drupal\graphql_webform\Enum\WebformElementDisplayOn;
use Drupal\graphql_webform\Enum\WebformElementHelpDisplay;
use Drupal\graphql_webform\Enum\WebformElementState;
use Drupal\graphql_webform\Enum\WebformElementTitleDisplay;
use Drupal\graphql_webform\Enum\WebformElementTrigger;
use Drupal\graphql_webform\Enum\WebformWeekday;
use Drupal\graphql_webform\Model\WebformSubmissionResult;
use Drupal\graphql_webform\Model\WebformSubmissionValidationError;
use Drupal\graphql_webform\WebformSchemaBuilder;
use Drupal\webform\Plugin\WebformElementManagerInterface;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * GraphQL schema extension for the Webform module.
 *
 * @SchemaExtension(
 *   id = "webform",
 *   name = "Webform",
 *   description = "Provides types and fields related to webforms.",
 *   schema = "webform"
 * )
 */
class WebformExtension extends SdlSchemaExtensionPluginBase {

  /**
   * A list of element IDs exposed by the Webform element plugin manager.
   *
   * Do not access this property directly. Use ::getElementIds() instead.
   *
   * @var string[]
   */
  protected array $elementIds = [];

  /**
   * Constructs a new WebformExtension GraphQL schema extension plugin.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $pluginId
   *   The plugin ID.
   * @param array $pluginDefinition
   *   The plugin definition array.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\webform\Plugin\WebformElementManagerInterface $webformElementManager
   *   The Webform element plugin manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    array $pluginDefinition,
    ModuleHandlerInterface $moduleHandler,
    protected WebformElementManagerInterface $webformElementManager,
    protected ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition, $moduleHandler);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler'),
      $container->get('plugin.manager.webform.element'),
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   *
   * Composes the base schema definition for the Webform module from the
   * following sources:
   * - The base schema definition file (`webform.base.graphqls`).
   * - A schema definition containing all element types exposed by the Webform
   *   module.
   */
  public function getBaseDefinition(): string {
    $definitions = [parent::getBaseDefinition()];

    $schemaBuilder = new WebformSchemaBuilder();
    foreach ($this->webformElementManager->getDefinitions() as $id => $definition) {
      /** @var \Drupal\webform\Plugin\WebformElementBase $instance */
      $instance = $this->webformElementManager->createInstance($id, []);
      $description = (string) $definition['description'];
      $schemaBuilder->generateElementType($id, $description, $instance);
    }
    $schemaBuilder->generateElementType('unexposed', 'Unexposed webform element.');

    $definitions[] = $schemaBuilder->getGeneratedSchema();

    return implode("\n\n", $definitions);
  }

  /**
   * {@inheritdoc}
   */
  public function registerResolvers(ResolverRegistryInterface $registry): void {
    $builder = new ResolverBuilder();

    $registry->addTypeResolver('WebformElement', function ($element) {
      $plugin_id = match (TRUE) {
        // In a regular element, the type is stored in #webform_plugin_id.
        !empty($element['#webform_plugin_id']) => $element['#webform_plugin_id'],
        // Some composite elements (like 'address') are composed of standard
        // form elements rather than webform elements. The element ID is stored
        // in #type, sharing the same ID as the corresponding webform element.
        !empty($element['#webform_composite_id']) && !empty($element['#type']) => $element['#type'],
        default => NULL,
      };
      if ($plugin_id) {
        // If the base plugin ID is passed this means the Webform module did not
        // derive a more specific plugin for the element. This can happen e.g.
        // if a webform contains an orphaned element from a disabled module.
        if ($plugin_id === 'webform_element') {
          return 'WebformElementUnexposed';
        }
        if (in_array($plugin_id, $this->getElementIds())) {
          return 'WebformElement' . $this->toUpperCamelCase($plugin_id);
        }
      }

      return 'WebformElementUnexposed';
    });

    $registry->addFieldResolver('Query', 'webformById',
      $builder->produce('webform_load')
        ->map('id', $builder->fromArgument('id'))
        ->map('sourceEntityType', $builder->fromArgument('sourceEntityType'))
        ->map('sourceEntityId', $builder->fromArgument('sourceEntityId'))
    );

    $registry->addFieldResolver('Query', 'webformConfirmation', $builder->compose(
      $builder->produce('webform_load_submission')
        ->map('id', $builder->fromArgument('submissionId'))
        ->map('token', $builder->fromArgument('token')),
      $builder->produce('webform_submission_confirmation')
        ->map('submission', $builder->fromParent())
        ->map('webform', $builder->produce('webform_load')
          ->map('id', $builder->fromArgument('webformId'))
        )
    ));

    // Expose the metadata of the webform element.
    $registry->addFieldResolver('WebformElement', 'metadata', $builder->callback(
      static fn (array $value): array => $value
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'key', $builder->callback(
      // If the element key is not found, we might be dealing with a composite
      // element.
      static fn (array $value): string => $value['#webform_key'] ?? $value['#webform_composite_key'] ?? ''
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'type', $builder->callback(
      static fn (array $value): string => match (TRUE) {
        // In a regular element, the type is stored in the '#webform_plugin_id'.
        !empty($value['#webform_plugin_id']) => $value['#webform_plugin_id'],
        // In some composite elements normal Form API element types might be
        // used, and the type is stored in the '#type' property.
        !empty($value['#webform_composite_id']) && !empty($value['#type']) => $value['#type'],
        default => '',
      }
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'name', $builder->callback(
      static function ($value) {
        $name = $value['#name'] ?? '';
        // When an element can be multiple, we remove the nested name of the
        // input. That way multiple elements with the same name can be
        // submitted and will be parsed as multiple entries.
        return str_replace('[items][0][_item_]', '[0]', $name);
      }
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'flex', $builder->callback(
      static function ($value) {
        $flex = $value['#flex'] ?? 1;
        // When an element can be multiple, we remove the nested name of the
        // input. That way multiple elements with the same name can be
        // submitted and will be parsed as multiple entries.
        return $flex;
      }
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'element', $builder->callback(
      static fn (array $value): array => $value
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'defaultValue', $builder->callback(
      static fn (array $value) => $value['#default_value'] ?? NULL ? json_encode($value['#default_value']) : NULL
    ));

    $typeFields = [
      'WebformElementMetadata' => [
        'title' => 'string',
        'description' => 'string',
        'multiple' => 'int',
        'required' => 'boolean',
        'requiredError' => 'string',
        'helpTitle' => 'string',
        'help' => 'string',
        'moreTitle' => 'string',
        'more' => 'string',
        'fieldPrefix' => 'string',
        'fieldSuffix' => 'string',
        'disabled' => 'boolean',
      ],
      'WebformElementMarkupBase' => [
        'markup' => 'string',
      ],
      'WebformElementNumber' => [
        'min' => 'int',
        'max' => 'int',
        'step' => 'int',
      ],
      'WebformElementRange' => [
        'min' => 'int',
        'max' => 'int',
        'step' => 'int',
      ],
      'WebformElementTextBase' => [
        'readonly' => 'boolean',
        'size' => 'int',
        'minlength' => 'int',
        'maxlength' => 'int',
        'placeholder' => 'string',
        'autocomplete' => 'string',
      ],
      'WebformElementTextfield' => [
        'inputMask' => 'string',
      ],
      'WebformElementWebformRating' => [
        'max' => 'int',
      ],
      'WebformElementWebformScale' => [
        'min' => 'int',
        'max' => 'int',
      ],
      'WebformElementWebformSection' => [
        'titleTag' => 'string',
      ],
    ];

    foreach ($typeFields as $interface => $fields) {
      foreach ($fields as $field => $type) {
        $registry->addFieldResolver($interface, $field,
          $builder->produce('webform_element_property')
            ->map('element', $builder->fromParent())
            ->map('property', $builder->fromValue($field))
            ->map('type', $builder->fromValue($type))
        );
      }
    }

    $registry->addFieldResolver('WebformElementDateBase', 'min',
      $builder->produce('webform_date_extremum')
        ->map('element', $builder->fromParent())
        ->map('extremum', $builder->fromValue('min'))
        ->map('webform', $builder->fromContext('webform'))
    );

    $registry->addFieldResolver('WebformElementDateBase', 'max',
      $builder->produce('webform_date_extremum')
        ->map('element', $builder->fromParent())
        ->map('extremum', $builder->fromValue('max'))
        ->map('webform', $builder->fromContext('webform'))
    );

    $registry->addFieldResolver('WebformElementDateBase', 'allowedDays', $builder->callback(
      function (array $value): array {
        $days = $value['#date_days'] ?? range(0, 6);
        return array_map(fn ($day) => (WebformWeekday::tryFrom($day) ?? WebformWeekday::Monday)->toGraphQLEnumValue(), array_map('intval', $days));
      }
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'titleDisplay', $builder->callback(
        static fn (array $value) => (WebformElementTitleDisplay::tryFrom($value['#_title_display'] ?? 'default') ?? WebformElementTitleDisplay::Default)->toGraphQLEnumValue()
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'states', $builder->callback(
      static fn (array $value) => $value['#_webform_states'] ?? []
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'descriptionDisplay', $builder->callback(
      static fn (array $value) => (WebformElementDescriptionDisplay::tryFrom($value['#description_display'] ?? 'default') ?? WebformElementDescriptionDisplay::Default)->toGraphQLEnumValue()
    ));

    $registry->addFieldResolver('WebformElementMetadata', 'helpDisplay', $builder->callback(
      static fn (array $value) => (WebformElementHelpDisplay::tryFrom($value['#help_display'] ?? 'default') ?? WebformElementHelpDisplay::Default)->toGraphQLEnumValue()
    ));

    $registry->addFieldResolver('WebformElementMarkupBase', 'displayOn', $builder->callback(
      static fn (array $value) => (WebformElementDisplayOn::tryFrom($value['#display_on'] ?? '') ?? WebformElementDisplayOn::DisplayOnForm)->toGraphQLEnumValue()
    ));

    // WebformElementProcessedText markup.
    $registry->addFieldResolver('WebformElementProcessedText', 'markup',
      $builder->produce('webform_render_element')
        ->map('element', $builder->fromParent())
    );

    $registry->addFieldResolver('WebformElementWebformAddress', 'items', $builder->compose(
      $builder->callback(function ($value) {
        $elementKeys = array_keys($value['#webform_composite_elements']);
        $elements = [];

        foreach ($elementKeys as $key) {
          if (!empty($value[$key])) {
            $elements[] = $value[$key];
          }
        }
        return $elements;
      })
    ));

    // Resolve the child elements of elements that contain other elements.
    $registry->addFieldResolver('WebformElementContainerBase', 'elements', $builder->compose(
      // The ResolverBuilder doesn't seem to have a method to allow a resolver
      // to act on the value itself, so we construct a fake "parent" that
      // contains the value and pass it using ::fromParent().
      $builder->callback(static fn ($value) => $value),
      $builder->produce('webform_elements')
        ->map('parent', $builder->fromParent())
        ->map('webform', $builder->fromContext('webform'))
    ));

    // Expose information about elements that accept multiple values.
    $registry->addFieldResolver('WebformElementMultipleValuesBase', 'multipleValues',
      $builder->produce('webform_element_multiple_values')
        ->map('element', $builder->fromParent())
    );

    // Resolve the options for checkboxes, radio buttons, etc.
    $registry->addFieldResolver('WebformElementOptionsBase', 'options',
      $builder->produce('webform_element_options')
        ->map('element', $builder->fromParent())
    );

    // Resolve the empty option for select elements.
    $registry->addFieldResolver('WebformElementSelect', 'emptyOption', $builder->callback(
      static fn (array $value): ?array => !empty($value['#empty_option']) ? ['value' => $value['#empty_value'] ?? '', 'title' => $value['#empty_option']] : NULL
    ));

    // Resolver for the number of rows of a textarea.
    $registry->addFieldResolver('WebformElementTextarea', 'rows', $builder->callback(
      // The default number of rows is 3.
      // @see \Drupal\webform\Plugin\WebformElement\Textarea::form()
      static fn (array $value): int => (int) ($value['#rows'] ?? 3)
    ));

    // Resolve regex validation patterns on text based elements.
    $registry->addFieldResolver('WebformElementTextBase', 'pattern',
      $builder->produce('webform_element_validation_pattern')
        ->map('element', $builder->fromParent())
    );

    // Resolve the submit label of the WebformAction element.
    $registry->addFieldResolver('WebformElementWebformActions', 'submitLabel', $builder->callback(
      static fn (array $value): string => $value['#submit__label'] ?? ''
    ));

    // Resolve the properties of the WebformElementWebformFlexbox element.
    $registry->addFieldResolver('WebformElementWebformFlexbox', 'alignItems', $builder->callback(
      static fn (array $value): string => (string) ($value['#align_items'] ?? 'flex-start')
    ));

    // Resolve the options for the term select element.
    $registry->addFieldResolver('WebformElementWebformTermSelect', 'options',
      $builder->produce('webform_term_select_options')
        ->map('element', $builder->fromParent())
        ->map('depth', $builder->fromArgument('depth'))
    );

    $this->addWebformFields($registry, $builder);
    $this->addMutationFields($registry, $builder);
    $this->registerStateFields($registry, $builder);

    // Resolve the max filesize of managed file elements.
    $registry->addFieldResolver('WebformElementManagedFileBase', 'maxFilesize',
      $builder->callback(function (array $value): int {
        // Check if the webform element has a max filesize set. If set this is
        // representing the size in MB.
        $size = $value['#max_filesize'] ?? NULL;
        if (!empty($size)) {
          $size .= 'MB';
        }

        // Consider the other possible sources for the max filesize: the PHP
        // limit and the (optional) default webform setting.
        $sizes = array_filter([
          $size,
          Environment::getUploadMaxSize(),
          $this->getWebformSetting('file.default_max_filesize') ?: Environment::getUploadMaxSize(),
        ]);

        // Convert to bytes.
        $sizes = array_map(fn ($size): int => (int) Bytes::toNumber($size), $sizes);

        // Return the smallest value.
        return min($sizes);
      })
    );

    // Resolve the file extensions of managed file elements.
    $registry->addFieldResolver('WebformElementManagedFileBase', 'fileExtensions',
      $builder->callback(function (array $value): ?string {
        $extensions = $value['#file_extensions'] ?? NULL;
        if (!empty($extensions)) {
          return (string) $extensions;
        }
        $file_type = str_replace('webform_', '', $value['#type']);
        return $this->getWebformSetting("file.default_{$file_type}_extensions");
      })
    );
  }

  /**
   * Registers the basic webform fields.
   *
   * @param \Drupal\graphql\GraphQL\ResolverRegistryInterface $registry
   *   The resolver registry.
   * @param \Drupal\graphql\GraphQL\ResolverBuilder $builder
   *   The resolver builder.
   */
  protected function addWebformFields(ResolverRegistryInterface $registry, ResolverBuilder $builder): void {
    // Provide the fields for the Webform type.
    $registry->addFieldResolver('Webform', 'id',
      $builder->produce('entity_id')
        ->map('entity', $builder->fromParent())
    );

    // The form title which takes into account the source entity.
    $registry->addFieldResolver('Webform', 'title',
      $builder->produce('webform_title')
        ->map('webform', $builder->fromParent())
        ->map('sourceEntity', $builder->produce('entity_load')
          ->map('type', $builder->fromContext('webform_source_entity_type'))
          ->map('id', $builder->fromContext('webform_source_entity_id'))
          ->map('accessOperation', $builder->fromValue('view'))
        )
    );

    // The label of the webform entity.
    $registry->addFieldResolver('Webform', 'label',
      $builder->produce('entity_label')->map('entity', $builder->fromParent())
    );

    // The webform elements.
    $registry->addFieldResolver('Webform', 'elements', $builder->compose(
      $builder->context('webform', $builder->fromParent()),
      $builder->produce('webform_elements')->map('webform', $builder->fromParent())
    ));

    // The source entity type, ID and label.
    $registry->addFieldResolver('Webform', 'sourceEntityType', $builder->fromContext('webform_source_entity_type'));
    $registry->addFieldResolver('Webform', 'sourceEntityId', $builder->fromContext('webform_source_entity_id'));
    $registry->addFieldResolver('Webform', 'sourceEntityLabel',
      $builder->produce('entity_label')
        ->map('entity', $builder->produce('entity_load')
          ->map('type', $builder->fromContext('webform_source_entity_type'))
          ->map('id', $builder->fromContext('webform_source_entity_id'))
          ->map('accessOperation', $builder->fromValue('view'))
        )
    );

    $registry->addFieldResolver('Webform', 'settings',
      $builder->produce('webform_settings')
        ->map('webform', $builder->fromParent())
    );
  }

  /**
   * Registers the fields related to mutations.
   *
   * @param \Drupal\graphql\GraphQL\ResolverRegistryInterface $registry
   *   The resolver registry.
   * @param \Drupal\graphql\GraphQL\ResolverBuilder $builder
   *   The resolver builder.
   */
  protected function addMutationFields(ResolverRegistryInterface $registry, ResolverBuilder $builder): void {
    // The webform submission mutation.
    $registry->addFieldResolver('Mutation', 'submitWebform', $builder->compose(
      $builder->produce('webform_submit')
        ->map('id', $builder->fromArgument('id'))
        ->map('elements', $builder->fromArgument('elements'))
        ->map('files', $builder->fromArgument('files'))
        ->map('sourceEntityId', $builder->fromArgument('sourceEntityId'))
        ->map('sourceEntityType', $builder->fromArgument('sourceEntityType'))
    ));

    $registry->addFieldResolver('WebformSubmissionResult', 'errors',
      $builder->callback(fn (WebformSubmissionResult $result): array => $result->getErrors())
    );

    $registry->addFieldResolver('WebformSubmissionResult', 'validationErrors',
      $builder->callback(fn (WebformSubmissionResult $result): array => $result->getValidationErrors())
    );

    $registry->addFieldResolver('WebformSubmissionResult', 'submission',
      $builder->callback(fn (WebformSubmissionResult $result): ?WebformSubmissionInterface => $result->getSubmission())
    );

    $registry->addFieldResolver('WebformSubmission', 'confirmation',
      $builder->produce('webform_submission_confirmation')
        ->map('submission', $builder->fromParent())
        ->map('webform', $builder->fromValue(NULL)
    ));

    $registry->addFieldResolver('WebformSubmission', 'id',
      // The id can be null if the submission saving is disabled on the webform.
      $builder->callback(fn (WebformSubmissionInterface $submission): ?int => $submission->id() !== NULL ? (int) $submission->id() : NULL)
    );

    $registry->addFieldResolver('WebformSubmission', 'sourceEntityType',
      $builder->callback(fn (WebformSubmissionInterface $submission): ?string => $submission->getSourceEntity()?->getEntityTypeId())
    );

    $registry->addFieldResolver('WebformSubmission', 'sourceEntityId',
      $builder->callback(fn (WebformSubmissionInterface $submission): ?string => match ($source_entity = $submission->getSourceEntity()) {
        NULL => NULL,
        default => (string) $source_entity->id(),
      })
    );

    $registry->addFieldResolver('WebformSubmission', 'token',
      $builder->callback(fn (WebformSubmissionInterface $submission): string => $submission->getToken())
    );

    $registry->addFieldResolver('WebformSubmission', 'tokenUrl',
      $builder->produce('webform_submission_token_url')
        ->map('submission', $builder->fromParent())
    );

    $registry->addFieldResolver('WebformSubmission', 'uuid',
      $builder->callback(fn (WebformSubmissionInterface $submission): string => $submission->uuid())
    );

    $registry->addFieldResolver('WebformSubmission', 'webform',
      $builder->callback(fn (WebformSubmissionInterface $submission): WebformInterface => $submission->getWebform())
    );

    $registry->addFieldResolver('WebformSubmissionValidationError', 'element',
      $builder->callback(fn (WebformSubmissionValidationError $result): ?string => $result->getElementId())
    );

    $registry->addFieldResolver('WebformSubmissionValidationError', 'messages',
      $builder->callback(fn (WebformSubmissionValidationError $result): array => $result->getMessages())
    );
  }

  /**
   * Register the webform element state fields.
   *
   * @param \Drupal\graphql\GraphQL\ResolverRegistryInterface $registry
   *   The resolver registry.
   * @param \Drupal\graphql\GraphQL\ResolverBuilder $builder
   *   The resolver builder.
   */
  protected function registerStateFields(ResolverRegistryInterface $registry, ResolverBuilder $builder): void {
    // Add the other field, which is a nested element.
    foreach (WebformElementState::cases() as $state) {
      $registry->addFieldResolver('WebformElementStates', $state->value,
        $builder->produce('webform_element_state')
          ->map('element', $builder->fromParent())
          ->map('state', $builder->fromValue($state->value))
      );
    }

    $registry->addFieldResolver('WebformElementState', 'conditions', $builder->compose(
      $builder->callback(function ($value) {
        return $value['conditions'];
      })
    ));

    $registry->addFieldResolver('WebformElementState', 'logic', $builder->compose(
      $builder->callback(function ($value) {
        return $value['logic'];
      })
    ));

    $registry->addFieldResolver('WebformElementStateCondition', 'field', $builder->compose(
      $builder->callback(function ($value) {
        $selector = $value['selector'] ?? '';
        $matches = [];
        preg_match('/:input\[name="(.*?)["[]/', $selector, $matches);
        return $matches[1] ?? NULL;
      })
    ));

    $registry->addFieldResolver('WebformElementStateCondition', 'fieldValue', $builder->compose(
      $builder->callback(function ($value) {
        $condition_value = $value['value'] ?? '';
        $selector = $value['selector'] ?? '';
        $matches = [];
        preg_match('/:input\[name=".*\[(.*)]"/', $selector, $matches);

        return $matches[1] ?? $condition_value;
      })
    ));

    $registry->addFieldResolver('WebformElementStateCondition', 'trigger', $builder->compose(
      $builder->callback(function ($value) {
        $trigger = $value['trigger'] ?? '';
        return WebformElementTrigger::from($trigger)->toGraphQLEnumValue();
      })
    ));

    $registry->addFieldResolver('WebformElementStateCondition', 'value', $builder->compose(
      $builder->callback(function ($value) {
        return $value['value'] ?? '';
      })
    ));
  }

  /**
   * Returns the IDs of elements exposed by the Webform element plugin manager.
   *
   * @return string[]
   *   The list of element IDs.
   */
  protected function getElementIds(): array {
    if (empty($this->elementIds)) {
      $this->elementIds = array_keys($this->webformElementManager->getDefinitions());
    }
    return $this->elementIds;
  }

  /**
   * Converts the given string to upper camel case.
   *
   * Dashes and underscores are removed.
   *
   * @param string $string
   *   The string to convert.
   *
   * @return string
   *   The converted string.
   */
  protected function toUpperCamelCase(string $string): string {
    return str_replace(' ', '', ucwords(str_replace(['-', '_'], ' ', $string)));
  }

  /**
   * Returns the value of a setting from the webform module.
   *
   * @param string $name
   *   The name of the setting to retrieve.
   *
   * @return mixed
   *   The value of the setting.
   */
  protected function getWebformSetting(string $name): mixed {
    return $this->configFactory->get('webform.settings')->get($name);
  }

}
