<?php

declare(strict_types=1);

namespace Drupal\graphql_webform;

use Drupal\graphql_webform\Enum\ExposedSettings;
use Drupal\graphql_webform\Enum\WebformElementDescriptionDisplay;
use Drupal\graphql_webform\Enum\WebformElementDisplayOn;
use Drupal\graphql_webform\Enum\WebformElementHelpDisplay;
use Drupal\graphql_webform\Enum\WebformElementState;
use Drupal\graphql_webform\Enum\WebformElementTitleDisplay;
use Drupal\graphql_webform\Enum\WebformElementTrigger;
use Drupal\graphql_webform\Enum\WebformSubmissionConfirmationType;
use Drupal\graphql_webform\Enum\WebformWeekday;
use Drupal\webform\Element\WebformElementStates;
use Drupal\webform\Entity\Webform;
use Drupal\webform\Plugin\WebformElement\ContainerBase;
use Drupal\webform\Plugin\WebformElement\DateBase;
use Drupal\webform\Plugin\WebformElement\OptionsBase;
use Drupal\webform\Plugin\WebformElement\TextBase;
use Drupal\webform\Plugin\WebformElement\WebformManagedFileBase;
use Drupal\webform\Plugin\WebformElement\WebformMarkupBase;
use Drupal\webform\Plugin\WebformElement\WebformTermSelect;
use Drupal\webform\Plugin\WebformElementInterface;
use GraphQL\Type\Definition\EnumType;
use GraphQL\Type\Definition\InterfaceType;
use GraphQL\Type\Definition\ObjectType;
use GraphQL\Type\Definition\Type;
use GraphQL\Type\Schema;
use GraphQL\Utils\SchemaPrinter;

/**
 * Builds the GraphQL schema for the Webform module.
 */
class WebformSchemaBuilder {

  /**
   * The generated types.
   *
   * @var \GraphQL\Type\Definition\TypeWithFields[]
   */
  protected array $types;

  /**
   * The WebformElementContainerBase interface fields.
   */
  protected array $elementContainerBase;

  /**
   * The WebformElementDateBase interface fields.
   *
   * @var array
   */
  protected array $elementDateBase;

  /**
   * The WebformElementManagedFileBase interface fields.
   */
  protected array $elementManagedFileBase;

  /**
   * The WebformElementTextBase interface fields.
   */
  protected array $elementTextBase;

  public function __construct() {
    $this->types = [];

    $this->createStatesTypes();

    $this->types['WebformElementTitleDisplay'] = new EnumType([
      'name' => 'WebformElementTitleDisplay',
      'values' => WebformElementTitleDisplay::toArray(),
      'description' => 'The `WebformElementTitleDisplay` enum contains the placement options for the webform element title.',
    ]);

    $this->types['WebformElementDescriptionDisplay'] = new EnumType([
      'name' => 'WebformElementDescriptionDisplay',
      'values' => WebformElementDescriptionDisplay::toArray(),
      'description' => 'The `WebformElementDescriptionDisplay` enum contains the placement options for the webform element description.',
    ]);

    $this->types['WebformElementHelpDisplay'] = new EnumType([
      'name' => 'WebformElementHelpDisplay',
      'values' => WebformElementHelpDisplay::toArray(),
      'description' => 'The `WebformElementHelpDisplay` enum contains the placement options for the webform element help.',
    ]);

    $this->types['WebformWeekdays'] = new EnumType([
      'name' => 'WebformWeekdays',
      'values' => WebformWeekday::toArray(),
      'description' => 'The days of the week.',
    ]);

    $this->types['WebformElementMetadata'] = new ObjectType([
      'name' => 'WebformElementMetadata',
      'fields' => [
        'key' => $this->type(Type::nonNull(Type::string()), 'The key of the element.'),
        'type' => $this->type(Type::nonNull(Type::string()), 'The element type.'),
        'title' => $this->type(Type::string(), ''),
        'flex' => $this->type(Type::int(), 'The flex property specifies the length of the item, relative to the rest of the flexible items inside the same container.'),
        'description' => $this->type(Type::string(), ''),
        'multiple' => $this->type(Type::int(), 'The number of values that can be entered into the element. Unlimited if 0.'),
        'required' => $this->type(Type::nonNull(Type::boolean()), 'True if the user must enter a value.'),
        'requiredError' => $this->type(Type::string(), 'If set this message will be used when a required webform element is empty instead of the default "x field is required" message.'),
        'helpTitle' => $this->type(Type::string(), ''),
        'help' => $this->type(Type::string(), ''),
        'moreTitle' => $this->type(Type::string(), ''),
        'more' => $this->type(Type::string(), ''),
        'titleDisplay' => $this->types['WebformElementTitleDisplay'],
        'descriptionDisplay' => $this->types['WebformElementDescriptionDisplay'],
        'helpDisplay' => $this->types['WebformElementHelpDisplay'],
        'fieldPrefix' => $this->type(Type::string(), ''),
        'fieldSuffix' => $this->type(Type::string(), ''),
        'disabled' => $this->type(Type::boolean(), ''),
        'states' => $this->types['WebformElementStates'],
        'defaultValue' => $this->type(Type::string(), 'The default value to be used for the element. Since the element may support multiple values of different types, this is returned as a JSON encoded string.'),
      ],
    ]);

    $this->types['WebformElement'] = new InterfaceType([
      'name' => 'WebformElement',
      'fields' => $this->getElementDefinition(),
    ]);

    $this->elementContainerBase = [
      'elements' => fn () => Type::nonNull(Type::listOf(Type::nonNull($this->types['WebformElement']))),
    ];

    $this->types['WebformElementContainerBase'] = new InterfaceType([
      'name' => 'WebformElementContainerBase',
      'fields' => $this->elementContainerBase,
    ]);

    $this->types['WebformElementMultipleValues'] = new ObjectType([
      'name' => 'WebformElementMultipleValues',
      'fields' => [
        'limit' => [
          'type' => Type::int(),
          'description' => 'The maximum number of values that can be entered. If set to -1, an unlimited number of values can be entered.',
        ],
        'message' => [
          'type' => Type::string(),
          'description' => 'The error message to display when the maximum number of values is exceeded.',
        ],
        'headerLabel' => [
          'type' => Type::string(),
          'description' => 'The label for the header of the multiple values fieldset.',
        ],
        'minItems' => [
          'type' => Type::int(),
          'description' => 'The minimum number of items that must be entered.',
        ],
        'emptyItems' => [
          'type' => Type::int(),
          'description' => 'The number of empty items to display.',
        ],
        'addMore' => [
          'type' => Type::boolean(),
          'description' => 'Whether or not to display the "Add more" button.',
        ],
        'addMoreItems' => [
          'type' => Type::int(),
          'description' => 'The number of items to add when the "Add more" button is clicked.',
        ],
        'addMoreButtonLabel' => [
          'type' => Type::string(),
          'description' => 'The label for the "Add more" button.',
        ],
        'addMoreInput' => [
          'type' => Type::boolean(),
          'description' => 'Allow users to input the number of items to be added.',
        ],
        'addMoreInputLabel' => [
          'type' => Type::string(),
          'description' => 'The label for the input field that allows users to input the number of items to be added.',
        ],
        'itemLabel' => [
          'type' => Type::string(),
          'description' => 'The label for each item.',
        ],
        'noItemsMessage' => [
          'type' => Type::string(),
          'description' => 'The message to display when there are no items.',
        ],
        'sorting' => [
          'type' => Type::boolean(),
          'description' => 'Allow users to sort elements.',
        ],
        'operations' => [
          'type' => Type::boolean(),
          'description' => 'Allow users to add/remove elements.',
        ],
        'add' => [
          'type' => Type::boolean(),
          'description' => 'Whether to show the "Add" button.',
        ],
        'remove' => [
          'type' => Type::boolean(),
          'description' => 'Whether to show the "Remove" button.',
        ],
      ],
    ]);

    $this->types['WebformElementMultipleValuesBase'] = new InterfaceType([
      'name' => 'WebformElementMultipleValuesBase',
      'fields' => [
        'multipleValues' => $this->types['WebformElementMultipleValues'],
      ],
    ]);

    $this->types['WebformElementOption'] = new ObjectType([
      'name' => 'WebformElementOption',
      'fields' => [
        'value' => Type::nonNull(Type::string()),
        'description' => Type::string(),
        'title' => Type::nonNull(Type::string()),
      ],
    ]);

    $this->types['WebformElementOptionsBase'] = new InterfaceType([
      'name' => 'WebformElementOptionsBase',
      'fields' => [
        'options' => fn () => Type::listOf($this->types['WebformElementOption']),
      ],
    ]);

    $this->types['WebformElementDisplayOn'] = new EnumType([
      'name' => 'WebformElementDisplayOn',
      'values' => WebformElementDisplayOn::toArray(),
      'description' => 'The `WebformElementDisplayOn` Enum contains the various locations where the associated Webform element can be displayed: during form editing, form viewing, both, or not visible at all.',
    ]);

    $this->types['WebformSubmissionConfirmationType'] = new EnumType([
      'name' => 'WebformSubmissionConfirmationType',
      'values' => WebformSubmissionConfirmationType::toArray(),
      'description' => 'The `WebformSubmissionConfirmationType` Enum contains the various ways the confirmation message can be handled that is displayed after a Webform is submitted.',
    ]);

    $this->types['WebformElementMarkupBase'] = new InterfaceType([
      'name' => 'WebformElementMarkupBase',
      'fields' => $this->getMarkupBaseFields(),
    ]);

    $this->elementDateBase = [
      'min' => [
        'type' => Type::string(),
        'description' => 'The minimum date that can be selected, in "YYYY-MM-DD" format.',
      ],
      'max' => [
        'type' => Type::string(),
        'description' => 'The maximum date that can be selected, in "YYYY-MM-DD" format.',
      ],
      'allowedDays' => [
        'type' => Type::listOf($this->types['WebformWeekdays']),
        'description' => 'The days of the week that are allowed to be selected.',
      ],
    ];

    $this->types['WebformElementDateBase'] = new InterfaceType([
      'name' => 'WebformElementDateBase',
      'fields' => $this->elementDateBase,
    ]);

    $this->elementManagedFileBase = [
      'maxFilesize' => Type::int(),
      'fileExtensions' => Type::string(),
    ];

    $this->types['WebformElementManagedFileBase'] = new InterfaceType([
      'name' => 'WebformElementManagedFileBase',
      'fields' => $this->elementManagedFileBase,
    ]);

    // A type representing a Regex validation pattern with an error message.
    $this->types['WebformElementValidationPattern'] = new ObjectType([
      'name' => 'WebformElementValidationPattern',
      'fields' => [
        'rule' => Type::string(),
        'message' => Type::string(),
      ],
    ]);

    $this->elementTextBase = [
      'readonly' => Type::boolean(),
      'size' => Type::int(),
      'minlength' => Type::int(),
      'maxlength' => Type::int(),
      'placeholder' => Type::string(),
      'autocomplete' => Type::string(),
      'pattern' => $this->types['WebformElementValidationPattern'],
    ];

    $this->types['WebformElementTextBase'] = new InterfaceType([
      'name' => 'WebformElementTextBase',
      'fields' => $this->elementTextBase,
    ]);

    $this->createWebformSettingsType();
  }

  /**
   * Creates states related types.
   */
  protected function createStatesTypes() {
    $this->types['WebformStateLogic'] = new EnumType([
      'name' => 'WebformStateLogic',
      'values' => [
        'AND' => 'and',
        'OR' => 'or',
        'XOR' => 'xor',
      ],
      'description' => 'The logical operator used to join conditions together.',
    ]);

    $elementStates = WebformElementStates::getTriggerOptions();
    $triggersValues = [];
    foreach ($elementStates as $id => $label) {
      $triggersValues[WebformElementTrigger::from($id)->toGraphQlEnumValue()] = [
        'value' => $id,
        'description' => (string) $label,
      ];
    }
    $this->types['WebformStateConditionTrigger'] = new EnumType([
      'name' => 'WebformStateConditionTrigger',
      'values' => $triggersValues,
      'description' => 'The triggering event of the state.',
    ]);

    $this->types['WebformElementStateCondition'] = new ObjectType([
      'name' => 'WebformElementStateCondition',
      'fields' => [
        'field' => $this->type(Type::string(), ''),
        'fieldValue' => $this->type(Type::string(), ''),
        'value' => $this->type(Type::string(), ''),
        'trigger' => $this->types['WebformStateConditionTrigger'],
      ],
      'description' => 'The state condition.',
    ]);

    $this->types['WebformElementState'] = new ObjectType([
      'name' => 'WebformElementState',
      'fields' => [
        'conditions' => $this->type(Type::listOf($this->types['WebformElementStateCondition']), ''),
        'logic' => $this->types['WebformStateLogic'],
      ],
      'description' => 'An element state consisting of the conditions and the logical operator joining them.',
    ]);

    $stateOptions = WebformElementStates::getStateOptions();
    $stateOptionsMapped = [];
    foreach ($stateOptions as $group => $states) {
      foreach ($states as $state => $label) {
        $stateOptionsMapped[$state] = $group . ': ' . $label;
      }
    }
    $stateFields = [];
    foreach (WebformElementState::cases() as $state) {
      $stateFields[$state->value] = $this->type($this->types['WebformElementState'], $stateOptionsMapped[$state->toWebformStateId()]);
    }
    $this->types['WebformElementStates'] = new ObjectType([
      'name' => 'WebformElementStates',
      'fields' => $stateFields,
      'description' => 'All states of the element.',
    ]);
  }

  /**
   * Creates the WebformSettings type.
   */
  protected function createWebformSettingsType() {
    $default_settings = Webform::getDefaultSettings();

    // Some settings have a default value of NULL, so we need to map these
    // manually.
    $null_settings = [
      'ajax_speed' => 'integer',
      'limit_total' => 'integer',
      'limit_total_interval' => 'integer',
      'limit_user' => 'integer',
      'limit_user_interval' => 'integer',
      'entity_limit_total' => 'integer',
      'entity_limit_total_interval' => 'integer',
      'entity_limit_user' => 'integer',
      'entity_limit_user_interval' => 'integer',
      'purge_days' => 'integer',
    ];

    $fields = [];

    foreach ($default_settings as $setting => $value) {
      if (!ExposedSettings::tryFrom($setting)) {
        continue;
      }
      $type = $null_settings[$setting] ?? $value;
      $graphqlType = $this->getType($type);
      if ($graphqlType) {
        $name = CaseConverter::toCamelCase($setting);
        $fields[$name] = $graphqlType;
      }
    }

    $this->types['WebformSettings'] = new ObjectType([
      'name' => 'WebformSettings',
      'fields' => $fields,
    ]);
  }

  /**
   * Get the GraphQL type for a PHP type.
   *
   * @var mixed $value
   *   The value to get the type for.
   *
   * @return string|null
   *   The GraphQL type.
   */
  protected static function getType($value) {
    switch (gettype($value)) {
      case 'boolean':
        return Type::boolean();

      case 'integer':
        return Type::int();

      case 'double':
        return Type::float();

      // Arrays are serialized.
      case 'array':
      case 'string':
        return Type::string();
    }

    return NULL;
  }

  /**
   * Returns the GraphQL schema describing the Webform data types.
   *
   * @return string
   *   The generated schema.
   */
  public function getGeneratedSchema(): string {
    $schema = new Schema(['types' => $this->types]);

    return SchemaPrinter::doPrint($schema);
  }

  /**
   * Generate the types for the form elements.
   */
  public function generateElementType(string $id, string $description, ?WebformElementInterface $plugin = NULL): void {
    $interfaces = [fn () => $this->types['WebformElement']];
    $fields = $this->getElementDefinition();

    // Expose child elements on elements that have them.
    $elementIsContainer =
      // Containers: fieldsets, flexbox elements, etc.
      $plugin instanceof ContainerBase ||
      // Elements that are composed of multiple fields, e.g. address fields.
      $plugin?->isComposite();

    if ($elementIsContainer) {
      $interfaces[] = fn () => $this->types['WebformElementContainerBase'];
      $fields = [
        ...$fields,
        ...$this->elementContainerBase,
      ];
    }

    if ($plugin instanceof OptionsBase) {
      $interfaces[] = fn () => $this->types['WebformElementOptionsBase'];
      $fields = [
        ...$fields,
        'options' => fn () => Type::listOf($this->types['WebformElementOption']),
      ];
    }

    if ($plugin instanceof DateBase) {
      $interfaces[] = fn () => $this->types['WebformElementDateBase'];
      $fields = array_merge($fields, $this->elementDateBase);
    }

    if ($plugin instanceof WebformElementInterface && $plugin->supportsMultipleValues()) {
      $interfaces[] = fn () => $this->types['WebformElementMultipleValuesBase'];
      $fields = [
        ...$fields,
        'multipleValues' => fn () => $this->types['WebformElementMultipleValues'],
      ];
    }

    // The Webform element for selecting taxonomy terms has an option to limit
    // the depth of terms to choose from. Allow this to be overridden in the
    // GraphQL query.
    if ($plugin instanceof WebformTermSelect) {
      $fields['options'] = [
        'type' => Type::listOf($this->types['WebformElementOption']),
        'description' => 'The taxonomy terms to show in the select element.',
        'args' => [
          'depth' => [
            'type' => Type::int(),
            'description' => 'The depth of terms to show in the select element. If omitted this will use the depth set in the Webform element configuration.',
          ],
        ],
      ];
    }

    if ($plugin instanceof WebformMarkupBase) {
      $interfaces[] = fn () => $this->types['WebformElementMarkupBase'];
      $fields = array_merge($fields, $this->getMarkupBaseFields());
    }

    if ($plugin instanceof WebformManagedFileBase) {
      $interfaces[] = fn () => $this->types['WebformElementManagedFileBase'];
      $fields = array_merge($fields, $this->elementManagedFileBase);
    }

    if ($plugin instanceof TextBase) {
      $interfaces[] = fn () => $this->types['WebformElementTextBase'];
      $fields = [
        ...$fields,
        ...$this->elementTextBase,
      ];
    }

    $type_name = 'WebformElement' . CaseConverter::toUpperCamelCase($id);
    $this->types[$type_name] = new ObjectType([
      'name' => $type_name,
      'fields' => $fields,
      'description' => $description,
      'interfaces' => $interfaces,
    ]);
  }

  /**
   * Formats the given type and description as an array.
   *
   * @param \GraphQL\Type\Definition\Type $type
   *   The GraphQL type.
   * @param string $description
   *   The description.
   *
   * @return array
   *   An array with the type and description.
   */
  protected static function type(Type $type, string $description): array {
    return [
      'type' => $type,
      'description' => $description,
    ];
  }

  /**
   * Returns the fields that are common to all markup elements.
   *
   * @return array
   *   The 'markup' and 'displayOn' fields.
   */
  protected function getMarkupBaseFields(): array {
    return [
      'markup' => Type::string(),
      'displayOn' => $this->types['WebformElementDisplayOn'],
    ];
  }

  /**
   * Returns the definition for a GraphQL element.
   *
   * @return array
   *   The element definition, containing the base fields.
   */
  protected function getElementDefinition(): array {
    return [
      'metadata' => fn () => $this->types['WebformElementMetadata'],
    ];
  }

}
