<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel\Element;

use Drupal\Core\StringTranslation\ByteSizeMarkup;
use Drupal\Tests\graphql_webform\Kernel\GraphQLWebformKernelTestBase;

/**
 * Tests for the WebformElementManagedFile type.
 *
 * @group graphql_webform
 */
class ManagedFileTest extends GraphQLWebformKernelTestBase {

  /**
   * Tests managed files.
   *
   * @dataProvider fileLimitsProvider
   */
  public function testManagedFile(string|int|null $webformGlobalFileLimit, int $expectedFileLimit, int $expectedAudioFileLimit): void {
    $this->config('webform.settings')
      ->set('file.default_max_filesize', $webformGlobalFileLimit)
      ->save();

    $human_readable_file_limit = (string) ByteSizeMarkup::create($expectedFileLimit);
    $human_readable_audio_file_limit = (string) ByteSizeMarkup::create($expectedAudioFileLimit);

    $query = $this->getQueryFromFile('managed_file.gql');
    $this->assertResults($query, ['id' => 'graphql_webform_test_form'], [
      'form' => [
        'title' => 'GraphQL Webform test form',
        'elements' => [
          6 => [
            '__typename' => 'WebformElementManagedFile',
            'maxFilesize' => $expectedFileLimit,
            'fileExtensions' => 'gif jpg png txt',
            'metadata' => [
              '__typename' => 'WebformElementMetadata',
              'key' => 'file_upload',
              'type' => 'managed_file',
              'title' => 'File upload',
              'description' => "Description<br />One file only.<br />$human_readable_file_limit limit.<br />Allowed types: gif jpg png txt.\n",
            ],
          ],
          11 => [
            '__typename' => 'WebformElementWebformAudioFile',
            'maxFilesize' => $expectedAudioFileLimit,
            'fileExtensions' => 'mp3 ogg wav',
            'metadata' => [
              '__typename' => 'WebformElementMetadata',
              'key' => 'audio_files',
              'type' => 'webform_audio_file',
              'title' => 'Audio files',
              'description' => "Maximum 2 files.<br />$human_readable_audio_file_limit limit.<br />Allowed types: mp3 ogg wav.\n",
            ],
          ],
        ],
      ],
    ], $this->defaultCacheMetaData());
  }

  /**
   * Provides a set of combinations of file limits.
   */
  public static function fileLimitsProvider(): array {
    return [
      'global limit is lowest' => [
        // Configure a global file limit of 80 KB in the webform settings.
        '80 KB',
        // The managed file upload cannot exceed the global limit of 80 KB since
        // this is the lowest limit.
        81920,
        // Same for the audio file upload.
        81920,
      ],
      'global limit in bytes' => [
        // Webform global file limit set to 3 MB.
        3145728,
        // The managed file upload is limited to 1 MB on element level.
        1048576,
        // The audio file upload cannot exceed the global webform limit of 3 MB.
        3145728,
      ],
      'global limit exceeding PHP limit' => [
        // Webform global file limit set to 8 MB.
        8388608,
        // The managed file upload is limited to 1 MB on element level.
        1048576,
        // The audio file upload cannot exceed the PHP limit of 4 MB.
        4194304,
      ],
      'empty global limit' => [
        // Webform global file limit not set.
        NULL,
        // The managed file upload is limited to 1 MB on element level.
        1048576,
        // The audio file upload cannot exceed the PHP limit of 4 MB.
        4194304,
      ],
    ];
  }

}

namespace Drupal\Component\Utility;

/**
 * Overrides the ini_get() function to mock a max file size of 4 MB.
 */
function ini_get($name) {
  switch ($name) {
    case 'upload_max_filesize':
      return '0';

    case 'post_max_size':
      return '4M';

    default:
      return '';
  }
}
