<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel\Element;

use Drupal\Tests\graphql_webform\Kernel\GraphQLWebformKernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\node\NodeInterface;
use Drupal\webform\Entity\Webform;
use Drupal\webform\WebformInterface;

/**
 * Tests for the WebformElementSelect type.
 *
 * @group graphql_webform
 */
class TitleTest extends GraphQLWebformKernelTestBase {

  /**
   * A test node serving as the source entity in which the webform is embedded.
   */
  protected NodeInterface $node;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('node_type');
    $this->installEntitySchema('node');

    // Create a test node.
    $node_type = NodeType::create(['type' => 'article', 'name' => 'Article']);
    $node_type->save();
    $this->node = Node::create(['type' => 'article', 'title' => 'Source entity title']);
    $this->node->save();
  }

  /**
   * Tests the webform title and label.
   *
   * @dataProvider titleDataProvider
   */
  public function testTitle(string $setting, string $expected): void {
    // Update the webform settings.
    $webform = Webform::load('graphql_webform_test_form');
    $webform
      ->setSetting('form_title', $setting)
      ->save();

    $query = $this->getQueryFromFile('title.gql');
    $this->assertResults($query, [
      'id' => 'graphql_webform_test_form',
      'sourceEntityType' => 'node',
      'sourceEntityId' => (string) $this->node->id(),
    ], [
      'form' => [
        'title' => $expected,
        'label' => $webform->label(),
        'sourceEntityType' => 'node',
        'sourceEntityId' => (string) $this->node->id(),
        'sourceEntityLabel' => (string) $this->node->label(),
      ],
      // The source entity should be present as a cacheable dependency.
    ], $this->defaultCacheMetaData()->addCacheableDependency($this->node));
  }

  /**
   * Data provider for ::testTitle().
   */
  public static function titleDataProvider(): array {
    return [
      [
        WebformInterface::TITLE_SOURCE_ENTITY,
        'Source entity title',
      ],
      [
        WebformInterface::TITLE_WEBFORM,
        'GraphQL Webform test form',
      ],
      [
        WebformInterface::TITLE_SOURCE_ENTITY_WEBFORM,
        'Source entity title: GraphQL Webform test form',
      ],
      [
        WebformInterface::TITLE_WEBFORM_SOURCE_ENTITY,
        'GraphQL Webform test form: Source entity title',
      ],
    ];
  }

  /**
   * Tests the webform title when no source entity is provided.
   *
   * @dataProvider titleWithoutSourceEntityDataProvider
   */
  public function testTitleWithoutSourceEntity(string $setting): void {
    // Update the webform settings.
    $webform = Webform::load('graphql_webform_test_form');
    $webform
      ->setSetting('form_title', WebformInterface::TITLE_SOURCE_ENTITY)
      ->save();

    $query = $this->getQueryFromFile('title.gql');

    // Test both without passing a source entity, and passing a non-existing
    // source entity.
    foreach (['no source entity', 'non-existing source entity'] as $sourceEntity) {
      $variables = ['id' => 'graphql_webform_test_form'];
      if ($sourceEntity === 'non-existing source entity') {
        $variables['sourceEntityType'] = 'node';
        $variables['sourceEntityId'] = 'non-existing node ID';
      }

      // If no source entity is provided, the cacheable metadata should only
      // contain the webform. If a non-existing source entity is provided, then
      // the list cache tags for the source entity type should be added because
      // the entity can be created later.
      $cacheMetaData = $this->defaultCacheMetaData();

      if ($sourceEntity === 'non-existing source entity') {
        $type = \Drupal::entityTypeManager()->getDefinition('node');
        /** @var \Drupal\Core\Entity\EntityTypeInterface $type */
        $tags = $type->getListCacheTags();
        $cacheMetaData->addCacheTags($tags);
      }

      $this->assertResults($query, $variables, [
        'form' => [
          'title' => 'GraphQL Webform test form',
          'label' => $webform->label(),
          'sourceEntityType' => $variables['sourceEntityType'] ?? NULL,
          'sourceEntityId' => $variables['sourceEntityId'] ?? NULL,
          'sourceEntityLabel' => NULL,
        ],
      ], $cacheMetaData);
    }
  }

  /**
   * Data provider for ::testTitleWithoutSourceEntity().
   */
  public static function titleWithoutSourceEntityDataProvider(): array {
    return [
      [WebformInterface::TITLE_SOURCE_ENTITY],
      [WebformInterface::TITLE_WEBFORM],
      [WebformInterface::TITLE_SOURCE_ENTITY_WEBFORM],
      [WebformInterface::TITLE_WEBFORM_SOURCE_ENTITY],
    ];
  }

}
