<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Tests\graphql\Kernel\GraphQLTestBase;
use Drupal\Tests\graphql\Traits\QueryResultAssertionTrait;
use Drupal\graphql\GraphQL\Execution\ExecutionResult;
use Drupal\graphql\Plugin\SchemaExtensionPluginManager;
use Drupal\graphql_webform\Plugin\GraphQL\SchemaExtension\WebformExtension;
use GraphQL\Error\Error;

/**
 * Base class for GraphQL Webform kernel tests.
 */
abstract class GraphQLWebformKernelTestBase extends GraphQLTestBase {

  use QueryResultAssertionTrait;

  /**
   * The root schema definition from the GraphQL module.
   *
   * Replicated here because the original is in a protected method.
   *
   * @see \Drupal\graphql\Plugin\GraphQL\Schema\ComposableSchema::getSchemaDefinition()
   */
  protected const ROOT_SCHEMA = <<<GQL
    type Schema {
      query: Query
    }

    type Query
  GQL;

  /**
   * The schema extension plugin manager.
   */
  protected SchemaExtensionPluginManager $schemaExtensionPluginManager;

  /**
   * The webform extension plugin.
   */
  protected WebformExtension $webformExtension;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'file',
    'graphql',
    'graphql_webform',
    'graphql_webform_test',
    'path_alias',
    'webform',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('path_alias');
    $this->installSchema('webform', ['webform']);
    $this->installConfig(['webform', 'graphql_webform_test']);

    // Register the webform extension.
    $this->schemaExtensionPluginManager = $this->container->get('plugin.manager.graphql.schema_extension');
    $this->webformExtension = $this->schemaExtensionPluginManager->createInstance('webform');
    $this->setUpSchema(self::ROOT_SCHEMA);
    $this->webformExtension->registerResolvers($this->registry);
  }

  /**
   * {@inheritdoc}
   */
  protected function assertResultErrors(ExecutionResult $result, array $expected): void {
    // Retrieve the list of error strings.
    $errors = array_map(function (Error $error) {
      return $error->getMessage();
    }, $result->errors);

    // Initialize the status.
    $unexpected = [];
    $matchCount = array_map(function () {
      return 0;
    }, array_flip($expected));

    // Iterate through error messages.
    // Collect unmatched errors and count pattern hits.
    while ($error = array_pop($errors)) {
      $match = FALSE;
      foreach ($expected as $pattern) {
        if (@preg_match($pattern, '') === FALSE) {
          $match = $match || $pattern == $error;
          $matchCount[$pattern]++;
        }
        else {
          $match = $match || preg_match($pattern, $error);
          $matchCount[$pattern]++;
        }
      }

      if (!$match) {
        $unexpected[] = $error;
      }
    }

    // Create a list of patterns that never matched.
    $missing = array_keys(array_filter($matchCount, function ($count) {
      return $count == 0;
    }));

    $this->assertEquals([], $missing, "Missing errors:\n* " . implode("\n* ", $missing));
    $this->assertEquals([], $unexpected, "Unexpected errors:\n* " . implode("\n* ", $unexpected));
  }

  /**
   * {@inheritdoc}
   */
  protected function assertResultData(ExecutionResult $result, $expected): void {
    $data = $result->toArray();
    // Filter out empty data sets from the form elements. In our test queries we
    // are using conditional fields. GraphQL returns empty objects for unmatched
    // fields. These are not important for the test results.
    if (!empty($data['data']['form']['elements'])) {
      $elements = &$data['data']['form']['elements'];
      $elements = array_filter($elements, function ($value) {
        return !is_object($value) || !empty((array) $value);
      });
    }

    $this->assertArrayHasKey('data', $data, 'No result data.');
    $this->assertEquals($expected, $data['data'], 'Unexpected query result.');
  }

  /**
   * {@inheritdoc}
   */
  protected function assertResultMetadata(ExecutionResult $result, CacheableMetadata $expected): void {
    $this->assertEquals($expected->getCacheMaxAge(), $result->getCacheMaxAge(), 'Unexpected cache max age.');

    $missingContexts = array_diff($expected->getCacheContexts(), $result->getCacheContexts());
    $this->assertEmpty($missingContexts, 'The following expected cache contexts are missing: ' . implode(', ', $missingContexts));

    $unexpectedContexts = array_diff($result->getCacheContexts(), $expected->getCacheContexts());
    $this->assertEmpty($unexpectedContexts, 'The following cache contexts were present but not expected: ' . implode(', ', $unexpectedContexts));

    $missingTags = array_diff($expected->getCacheTags(), $result->getCacheTags());
    $this->assertEmpty($missingTags, 'The following expected cache tags are missing: ' . implode(', ', $missingTags));

    $unexpectedTags = array_diff($result->getCacheTags(), $expected->getCacheTags());
    $this->assertEmpty($unexpectedTags, 'The following cache tags were present but not expected: ' . implode(', ', $unexpectedTags));
  }

  /**
   * {@inheritdoc}
   */
  protected function setUpSchema($schema, $id = 'test', array $values = []): void {
    $this->mockSchema($id, $schema, [$this->webformExtension]);
    $this->mockSchemaPluginManager($id);
    $this->createTestServer($id, '/graphql/' . $id, $values);

    $this->schemaPluginManager->method('createInstance')
      ->with($this->equalTo($id))
      ->willReturn($this->schema);

    $this->container->set('plugin.manager.graphql.schema', $this->schemaPluginManager);
  }

  /**
   * {@inheritdoc}
   */
  protected function defaultCacheTags(): array {
    return array_merge(parent::defaultCacheTags(), [
      'config:webform.settings',
      'config:webform.webform.graphql_webform_test_form',
      'webform:graphql_webform_test_form',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  protected function defaultCacheContexts(): array {
    return array_merge(parent::defaultCacheContexts(), ['languages:language_interface', 'request_format', 'url.path']);
  }

}
