<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel\Mutation;

use Drupal\graphql_webform\Enum\WebformSubmissionConfirmationType;

/**
 * Tests that a confirmation message is returned after a form submission.
 *
 * @group graphql_webform
 */
class FormSubmissionConfirmationTest extends FormSubmissionTestBase {

  /**
   * Tests submitting a form with various confirmation configurations.
   *
   * @param \Drupal\graphql_webform\Enum\WebformSubmissionConfirmationType $confirmationType
   *   The confirmation type.
   * @param string|null $confirmationTitle
   *   The confirmation title.
   * @param string|null $confirmationMessage
   *   The confirmation message.
   * @param string|null $confirmationUrl
   *   The confirmation URL.
   * @param array $expectedConfirmation
   *   The expected confirmation result.
   *
   * @dataProvider confirmationProvider
   */
  public function testFormSubmissionConfirmation(WebformSubmissionConfirmationType $confirmationType, ?string $confirmationTitle, ?string $confirmationMessage, ?string $confirmationUrl, array $expectedConfirmation): void {
    // Configure the webform with the provided confirmation settings.
    $webform = $this->webformStorage->load('graphql_webform_test_form');
    $webform->setSetting('confirmation_type', $confirmationType->value)
      ->setSetting('confirmation_title', $confirmationTitle)
      ->setSetting('confirmation_message', $confirmationMessage)
      ->setSetting('confirmation_url', $confirmationUrl)
      ->save();

    // Create a GraphQL query that includes the confirmation data.
    $query = $this->getQueryFromFile('submission_confirmation.gql');

    $variables = [
      'id' => 'graphql_webform_test_form',
      'elements' => [(object) ['element' => 'required_text_field', 'value' => 'A value.']],
    ];

    // Build the expected result with the confirmation data.
    $expected_result = [
      'submitWebform' => [
        'submission' => [
          'confirmation' => $expectedConfirmation,
        ],
      ],
    ];

    $expected_cache_metadata = $this->defaultCacheMetaData()->addCacheTags(['webform_submission:1']);

    $this->assertResults($query, $variables, $expected_result, $expected_cache_metadata);
    $this->assertWebformSubmissionCount(1, 'A valid submission is saved.');
  }

  /**
   * Provides data for the testFormSubmissionConfirmation test.
   *
   * @return array
   *   An array of test cases.
   */
  public static function confirmationProvider(): array {
    return [
      'page confirmation' => [
        'confirmationType' => WebformSubmissionConfirmationType::Page,
        'confirmationTitle' => 'Thank you for your submission',
        'confirmationMessage' => 'Your submission has been received.',
        'confirmationUrl' => '',
        'expectedConfirmation' => [
          'type' => WebformSubmissionConfirmationType::Page->toGraphQlEnumValue(),
          'title' => 'Thank you for your submission',
          'message' => 'Your submission has been received.',
          'redirectUrl' => '',
        ],
      ],
      'url confirmation' => [
        'confirmationType' => WebformSubmissionConfirmationType::Url,
        'confirmationTitle' => NULL,
        'confirmationMessage' => NULL,
        'confirmationUrl' => '/thank-you',
        'expectedConfirmation' => [
          'type' => WebformSubmissionConfirmationType::Url->toGraphQlEnumValue(),
          'title' => NULL,
          'message' => 'New submission added to GraphQL Webform test form.',
          'redirectUrl' => '/thank-you',
        ],
      ],
      'inline confirmation' => [
        'confirmationType' => WebformSubmissionConfirmationType::Inline,
        'confirmationTitle' => NULL,
        'confirmationMessage' => 'Thanks for submitting the form.',
        'confirmationUrl' => '',
        'expectedConfirmation' => [
          'type' => WebformSubmissionConfirmationType::Inline->toGraphQlEnumValue(),
          'title' => NULL,
          'message' => 'Thanks for submitting the form.',
          'redirectUrl' => '',
        ],
      ],
      'message confirmation' => [
        'confirmationType' => WebformSubmissionConfirmationType::Message,
        'confirmationTitle' => 'Form submitted',
        'confirmationMessage' => 'Thank you for your feedback.',
        'confirmationUrl' => '',
        'expectedConfirmation' => [
          'type' => WebformSubmissionConfirmationType::Message->toGraphQlEnumValue(),
          'title' => 'Form submitted',
          'message' => 'Thank you for your feedback.',
          'redirectUrl' => '',
        ],
      ],
      'none confirmation' => [
        'confirmationType' => WebformSubmissionConfirmationType::None,
        'confirmationTitle' => NULL,
        'confirmationMessage' => NULL,
        'confirmationUrl' => '',
        'expectedConfirmation' => [
          'type' => WebformSubmissionConfirmationType::None->toGraphQlEnumValue(),
          'title' => NULL,
          'message' => 'New submission added to GraphQL Webform test form.',
          'redirectUrl' => '',
        ],
      ],
      'token replacement in title and message' => [
        'confirmationType' => WebformSubmissionConfirmationType::Page,
        'confirmationTitle' => 'Thank you [webform_submission:values:required_text_field]',
        'confirmationMessage' => 'Your value was [webform_submission:values:required_text_field]',
        'confirmationUrl' => '',
        'expectedConfirmation' => [
          'type' => WebformSubmissionConfirmationType::Page->toGraphQlEnumValue(),
          'title' => 'Thank you prefixA value.suffix',
          'message' => 'Your value was prefixA value.suffix',
          'redirectUrl' => '',
        ],
      ],
      'external url' => [
        'confirmationType' => WebformSubmissionConfirmationType::Url,
        'confirmationTitle' => NULL,
        'confirmationMessage' => NULL,
        'confirmationUrl' => 'https://example.com/thank-you',
        'expectedConfirmation' => [
          'type' => WebformSubmissionConfirmationType::Url->toGraphQlEnumValue(),
          'title' => NULL,
          'message' => 'New submission added to GraphQL Webform test form.',
          'redirectUrl' => 'https://example.com/thank-you',
        ],
      ],
    ];
  }

}
