<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel\Mutation;

use Drupal\graphql\GraphQL\Execution\ExecutionResult;
use Drupal\webform\WebformInterface;

/**
 * Tests submission of a Webform.
 *
 * @group graphql_webform
 */
class FormSubmissionTest extends FormSubmissionTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
  ];

  /**
   * Tests submitting a form with an invalid form ID.
   */
  public function testSubmitInvalidFormId(): void {
    $query = $this->getQueryFromFile('webform_submission.gql');
    $variables = ['id' => 'invalid_form_id', 'elements' => []];
    $this->assertResults(
      $query,
      $variables,
      [
        'submitWebform' => [
          'errors' => ['Webform invalid_form_id does not exist.'],
          'validationErrors' => [],
          'submission' => NULL,
        ],
      ],
      // Since we do not have a valid webform to submit, only the basic cache
      // metadata is associated with the response.
      $this->baseCacheMetadata()
    );

    $this->assertWebformSubmissionCount(0, 'An invalid submission is not saved.');
  }

  /**
   * Tests that a form submission is rejected if the user does not have access.
   *
   * @param bool $authenticated_access_allowed
   *   Whether authenticated users are allowed to submit the form.
   *
   * @dataProvider submitAccessProvider
   */
  public function testSubmitAccess(bool $authenticated_access_allowed): void {
    if (!$authenticated_access_allowed) {
      // Update the form configuration to disallow authenticated access.
      $webform = $this->webformStorage->load('graphql_webform_test_form');
      $access_rules = $webform->getAccessRules();
      $access_rules['create']['roles'] = ['anonymous'];
      $webform->setAccessRules($access_rules)->save();
    }

    $query = $this->getQueryFromFile('webform_submission.gql');
    $variables = [
      'id' => 'graphql_webform_test_form',
      'elements' => [(object) ['element' => 'required_text_field', 'value' => 'A value.']],
    ];

    $expected_cache_metadata = $this->baseCacheMetadata()
      ->addCacheContexts($this->defaultCacheContexts())
      ->setCacheTags($this->defaultCacheTags());
    // Add the submission cache tags if the submission was successful.
    if ($authenticated_access_allowed) {
      $expected_cache_metadata->addCacheTags(['webform_submission:1']);
    }

    $expected_errors = $authenticated_access_allowed ? [] : ['Please log in to access this form.'];

    $expected_submission = $authenticated_access_allowed ? [
      'id' => 1,
      'webform' => [
        'id' => 'graphql_webform_test_form',
      ],
    ] : NULL;

    $this->assertResults($query, $variables, [
      'submitWebform' => [
        'errors' => $expected_errors,
        'validationErrors' => [],
        'submission' => $expected_submission,
      ],
    ], $expected_cache_metadata);

    $this->assertWebformSubmissionCount((int) $authenticated_access_allowed);
  }

  /**
   * Provides data for the testSubmitAccess test.
   *
   * @return array
   *   An array of test cases.
   */
  public static function submitAccessProvider(): array {
    return [
      'authenticated access allowed' => [TRUE],
      'authenticated access not allowed' => [FALSE],
    ];
  }

  /**
   * Tests submitting a form with an invalid value.
   */
  public function testValidationError(): void {
    $query = $this->getQueryFromFile('webform_submission.gql');
    $variables = [
      'id' => 'graphql_webform_test_form',
      'elements' => [(object) ['element' => 'required_text_field', 'value' => 'This value is too long.']],
    ];

    $expected_cache_metadata = $this->baseCacheMetadata()
      ->addCacheContexts($this->defaultCacheContexts())
      ->setCacheTags($this->defaultCacheTags());

    $this->assertResults($query, $variables, [
      'submitWebform' => [
        'errors' => [],
        'validationErrors' => [
          [
            'element' => 'required_text_field',
            'messages' => ['Required text field cannot be longer than 12 characters but is currently 23 characters long.'],
          ],
        ],
        'submission' => NULL,
      ],
    ], $expected_cache_metadata);

    $this->assertWebformSubmissionCount(0, 'An invalid submission is not saved.');
  }

  /**
   * Tests submitting a form that is closed.
   *
   * @param string $closed_message
   *   The closed message to set on the form.
   * @param string $default_closed_message
   *   The default closed message configured in the Webform global settings.
   * @param string $expected_error_message
   *   The expected error message.
   *
   * @dataProvider closedFormProvider
   */
  public function testClosedForm(string $closed_message, string $default_closed_message, string $expected_error_message): void {
    /** @var \Drupal\webform\WebformInterface $webform */
    $webform = $this->webformStorage->load('graphql_webform_test_form');
    $webform
      // Close the test form.
      ->setStatus(WebformInterface::STATUS_CLOSED)
      // Set the closed message.
      ->setSetting('form_close_message', $closed_message)
      ->save();

    // Set the default closed message.
    $this->config('webform.settings')
      ->set('settings.default_form_close_message', $default_closed_message)
      ->save();

    $query = $this->getQueryFromFile('webform_submission.gql');
    $variables = [
      'id' => 'graphql_webform_test_form',
      'elements' => [(object) ['element' => 'required_text_field', 'value' => 'A value.']],
    ];

    $this->assertResults($query, $variables, [
      'submitWebform' => [
        'errors' => [$expected_error_message],
        'validationErrors' => [],
        'submission' => NULL,
      ],
    ], $this->baseCacheMetadata()
      ->addCacheContexts($this->defaultCacheContexts())
      ->setCacheTags($this->defaultCacheTags()));

    $this->assertWebformSubmissionCount(0, 'A submission for a closed Webform is not saved.');
  }

  /**
   * Provides data for the testClosedForm test.
   *
   * @return array
   *   An array of test cases.
   */
  public static function closedFormProvider(): array {
    return [
      'custom message' => [
        '<p>This form is closed because of reasons.</p>',
        'Sorry… This form is closed to new submissions.',
        'This form is closed because of reasons.',
      ],
      'default message' => [
        '',
        'Sorry… This form is closed to new submissions.',
        'Sorry… This form is closed to new submissions.',
      ],
      'fallback message' => [
        '',
        '',
        'This form is closed to new submissions.',
      ],
    ];
  }

  /**
   * Tests submitting a form with a required field omitted.
   *
   * This should return the required text that was configured in the webform.
   */
  public function testOmitRequiredField(): void {
    $query = $this->getQueryFromFile('webform_submission.gql');
    $variables = [
      'id' => 'graphql_webform_test_form',
      'elements' => [],
    ];

    $this->assertResults($query, $variables, [
      'submitWebform' => [
        'errors' => [],
        'validationErrors' => [
          [
            'element' => 'required_text_field',
            'messages' => ['This field is required because it is important.'],
          ],
        ],
        'submission' => NULL,
      ],
    ], $this->baseCacheMetadata()
      ->addCacheContexts($this->defaultCacheContexts())
      ->setCacheTags($this->defaultCacheTags())
    );

    $this->assertWebformSubmissionCount(0, 'An invalid submission is not saved.');
  }

  /**
   * Tests a valid form submission.
   */
  public function testValidSubmission(): void {
    $query = $this->getQueryFromFile('webform_submission.gql');
    $variables = [
      'id' => 'graphql_webform_test_form',
      'elements' => [
        (object) [
          'element' => 'required_text_field',
          'value' => 'A value.',
        ],
        (object) [
          'element' => 'checkboxes',
          'value' => ['phobos', 'deimos'],
        ],
        (object) [
          'element' => 'custom_composite_container[0][first_name]',
          'value' => 'Harold',
        ],
        (object) [
          'element' => 'custom_composite_container[0][last_name]',
          'value' => 'Fon',
        ],
        (object) [
          'element' => 'address_using_default_configuration[given_name]',
          'value' => 'Jasmine',
        ],
        (object) [
          'element' => 'address_using_default_configuration[family_name]',
          'value' => 'Mary',
        ],
      ],
    ];

    $this->assertResults($query, $variables, [
      'submitWebform' => [
        'errors' => [],
        'validationErrors' => [],
        'submission' => [
          'id' => '1',
          'webform' => [
            'id' => 'graphql_webform_test_form',
          ],
        ],
      ],
    ], $this->defaultCacheMetaData()->addCacheTags(['webform_submission:1']));

    // There should be one submission in the database.
    $this->assertWebformSubmissionCount(1, 'A valid submission is saved.');

    // Load the webform submission and check it contains the expected value.
    $submissions = $this->webformSubmissionStorage->loadMultiple();
    /** @var \Drupal\webform\WebformSubmissionInterface $submission */
    $submission = reset($submissions);
    $this->assertEquals('A value.', $submission->getElementData('required_text_field'));
    $this->assertEquals(['phobos', 'deimos'], $submission->getElementData('checkboxes'));
    $this->assertEquals([['first_name' => 'Harold', 'last_name' => 'Fon']], $submission->getElementData('custom_composite_container'));
    $this->assertEquals(['given_name' => 'Jasmine', 'family_name' => 'Mary'], $submission->getElementData('address_using_default_configuration'));
  }

  /**
   * {@inheritdoc}
   */
  protected function assertResultData(ExecutionResult $result, $expected): void {
    // Enrich the expected result with values for the submission now that it has
    // been created.
    if ($id = $expected['submitWebform']['submission']['id'] ?? NULL) {
      /** @var \Drupal\webform\WebformSubmissionInterface $submission */
      $submission = $this->webformSubmissionStorage->load($id);
      $expected['submitWebform']['submission']['uuid'] = $submission->uuid();
      $expected['submitWebform']['submission']['token'] = $submission->getToken();
      $expected['submitWebform']['submission']['tokenUrl'] = $submission->getTokenUrl('view')->toString();
    }

    parent::assertResultData($result, $expected);
  }

}
