<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel\Mutation;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Tests\graphql_webform\Kernel\GraphQLWebformKernelTestBase;
use Drupal\webform\WebformEntityStorageInterface;
use Drupal\webform\WebformSubmissionStorageInterface;

/**
 * Base class for testing Webform submissions.
 */
abstract class FormSubmissionTestBase extends GraphQLWebformKernelTestBase {

  /**
   * The entity type manager.
   */
  protected ?EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Webform storage.
   */
  protected ?WebformEntityStorageInterface $webformStorage;

  /**
   * The Webform submission storage.
   */
  protected ?WebformSubmissionStorageInterface $webformSubmissionStorage;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    // The webform module requires the File module when a form allows file
    // uploads.
    'file',
    // The webform module depends silently on the Path Alias module, which is
    // fine since this is a required core module.
    'path_alias',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('path_alias');
    $this->installEntitySchema('webform_submission');
    $this->installConfig('webform');

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->webformStorage = $this->entityTypeManager->getStorage('webform');
    $this->webformSubmissionStorage = $this->entityTypeManager->getStorage('webform_submission');
  }

  /**
   * {@inheritdoc}
   */
  protected function defaultCacheContexts(): array {
    return ['user.permissions', 'languages:language_interface'];
  }

  /**
   * {@inheritdoc}
   */
  protected function defaultCacheMaxAge(): int {
    // Since these are mutations we don't want to cache the results. A
    // subsequent request will probably have different results.
    return 0;
  }

  /**
   * Returns the minimal cache metadata associated with any GraphQL response.
   *
   * @return \Drupal\Core\Cache\CacheableMetadata
   *   The cache metadata object.
   */
  protected function baseCacheMetadata(): CacheableMetadata {
    return (new CacheableMetadata())
      ->setCacheMaxAge($this->defaultCacheMaxAge())
      ->setCacheContexts(['user.permissions'])
      ->setCacheTags(['graphql_response', "config:graphql.graphql_servers.{$this->server->id()}"]);
  }

  /**
   * Checks the number of Webform submissions.
   *
   * @param int $expected_count
   *   The expected number of Webform submissions.
   * @param string $message
   *   The assertion message.
   */
  protected function assertWebformSubmissionCount(int $expected_count, string $message = ''): void {
    $count = $this->webformSubmissionStorage->getTotal();
    $this->assertEquals($expected_count, $count, $message);
  }

}
