<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel\Mutation;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\file\FileStorageInterface;
use Drupal\webform\Entity\WebformSubmission;
use Symfony\Component\HttpFoundation\Request;

/**
 * Test file uploads with GraphQL.
 *
 * @group graphql_webform
 */
final class FormSubmissionWithFileUploadTest extends FormSubmissionTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'file',
  ];

  /**
   * The file storage.
   */
  protected ?FileStorageInterface $fileStorage;

  /**
   * The file system.
   */
  protected ?FileSystemInterface $fileSystem;

  /**
   * An array of test files.
   */
  protected array $files = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);

    $this->fileStorage = $this->container->get('entity_type.manager')->getStorage('file');
    $this->fileSystem = $this->container->get('file_system');
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clean up files that were created during the test.
    foreach ($this->files as $file) {
      if (file_exists($file)) {
        unlink($file);
      }
    }

    parent::tearDown();
  }

  /**
   * {@inheritdoc}
   */
  public function register(ContainerBuilder $container): void {
    parent::register($container);

    // Register the private stream wrapper so we can test whether files can be
    // uploaded to the private filesystem as part of a webform submission.
    $container->register('stream_wrapper.private', 'Drupal\Core\StreamWrapper\PrivateStream')
      ->addTag('stream_wrapper', ['scheme' => 'private']);
  }

  /**
   * {@inheritdoc}
   */
  protected function setUpFilesystem(): void {
    // Set up the private filesystem in addition to the public filesystem.
    $public_file_directory = $this->siteDirectory . '/files';
    $private_file_directory = $this->siteDirectory . '/private';

    mkdir($this->siteDirectory, 0775);
    mkdir($this->siteDirectory . '/files', 0775);
    mkdir($this->siteDirectory . '/private', 0775);
    mkdir($this->siteDirectory . '/files/config/sync', 0775, TRUE);

    $this->setSetting('file_public_path', $public_file_directory);
    $this->setSetting('file_private_path', $private_file_directory);
    $this->setSetting('config_sync_directory', $this->siteDirectory . '/files/config/sync');
  }

  /**
   * Uploading a file using a wrong element name should return an error.
   */
  public function testUploadingFileToWrongElement(): void {
    $query = $this->getQueryFromFile('submission_with_file_upload.gql');
    $variables = [
      'elements' => [],
      'files' => [
        // Try to upload to a non-file upload element.
        ['element' => 'checkboxes', 'file' => NULL],
      ],
      'id' => 'graphql_webform_test_form',
    ];

    $this->assertResults($query, $variables, [
      'submitWebform' => [
        'errors' => [
          'Files cannot be uploaded to the "checkboxes" element since it is not a managed file element.',
        ],
        'validationErrors' => [
          [
            'element' => 'required_text_field',
            'messages' => [
              'This field is required because it is important.',
            ],
          ],
        ],
        'submission' => NULL,
      ],
    ]);
  }

  /**
   * Tests uploading a file as part of a webform submission.
   */
  public function testFileUpload(): void {
    // Create some test files to upload.
    foreach (['txt', 'mp3', 'ogg'] as $extension) {
      $file = $this->fileSystem->getTempDirectory() . '/graphql_webform_upload_test.' . $extension;
      // We are pretending to upload a file into temporary storage. Ensure the
      // file exists because the Symfony UploadedFile component will check that.
      touch($file);
      $this->files[$extension] = $file;
    }

    // Create a POST request with the files attached as multipart form data.
    $request = Request::create(
      uri: '/graphql/test',
      method: 'POST',
      parameters: [
        'query' => $this->getQueryFromFile('submission_with_file_upload.gql'),
        'variables' => [
          'elements' => [(object) ['element' => 'required_text_field', 'value' => 'A value.']],
          'files' => [
            // When using multipart form uploads, the parameter holding the file
            // has to be declared NULL.
            ['element' => 'file_upload', 'file' => NULL],
            ['element' => 'audio_files', 'file' => NULL],
            ['element' => 'audio_files', 'file' => NULL],
          ],
          'id' => 'graphql_webform_test_form',
        ],
        // The 'map' parameter is used to map attached files to variables.
        // @see https://github.com/jaydenseric/graphql-multipart-request-spec
        'map' => [
          '0' => ['variables.files.0.file'],
          '1' => ['variables.files.1.file'],
          '2' => ['variables.files.2.file'],
        ],
      ],
      files: [
        '0' => [
          'name' => 'test.txt',
          'type' => 'text/plain',
          'size' => 0,
          'tmp_name' => $this->files['txt'],
          'error' => UPLOAD_ERR_OK,
        ],
        '1' => [
          'name' => 'audio_file.mp3',
          'type' => 'audio/mpeg',
          'size' => 0,
          'tmp_name' => $this->files['mp3'],
          'error' => UPLOAD_ERR_OK,
        ],
        '2' => [
          'name' => 'audio_file.ogg',
          'type' => 'audio/ogg',
          'size' => 0,
          'tmp_name' => $this->files['ogg'],
          'error' => UPLOAD_ERR_OK,
        ],
      ],
    );

    $result = $this->executeMultiPartRequest($request);

    $this->assertNotEmpty($result->data->submitWebform ?? [], 'The response contains information about the submission.');
    $resultData = $result->data->submitWebform;

    $this->assertEmpty($resultData->errors, 'There are no errors.');
    $this->assertEmpty($resultData->validationErrors, 'There are no validation errors.');

    $submissionId = $resultData->submission->id;
    $this->assertIsInt($submissionId, 'A submission ID is returned.');

    // Check that the submission entity was created.
    $submission = WebformSubmission::load($submissionId);
    $this->assertInstanceOf(WebformSubmission::class, $submission, 'A submission entity was created.');

    // Check that the expected files are associated with the submission.
    $expectedUploads = [
      'file_upload' => [
        ['filename' => 'test.txt', 'mime' => 'text/plain'],
      ],
      'audio_files' => [
        ['filename' => 'audio_file.mp3', 'mime' => 'audio/mpeg'],
        ['filename' => 'audio_file.ogg', 'mime' => 'audio/ogg'],
      ],
    ];
    foreach ($expectedUploads as $elementName => $expectedFiles) {
      $returnedFileIds = (array) $submission->getElementData($elementName);
      $this->assertNotEmpty($returnedFileIds, 'One or more file IDs are associated with the webform submission.');
      $this->assertCount(count($expectedFiles), $returnedFileIds, 'The correct number of files is associated with the webform submission.');
      $expectedFilenames = array_column($expectedFiles, 'filename');
      $expectedMimeTypes = array_column($expectedFiles, 'mime');
      $expectedScheme = $elementName === 'file_upload' ? 'public' : 'private';
      foreach ($expectedFiles as $i => $expectedFile) {
        $file = File::load($returnedFileIds[$i]);
        $this->assertInstanceOf(FileInterface::class, $file, sprintf('A file entity was created for upload #%d of element "%s".', $i, $elementName));
        $this->assertEquals($expectedFilenames[$i], $file->getFilename(), sprintf('The file for upload #%d of element "%s" has the correct filename "%s".', $i, $elementName, $expectedFilenames[$i]));
        $this->assertEquals($expectedMimeTypes[$i], $file->getMimeType(), sprintf('The file for upload #%d of element "%s" has the correct MIME type "%s".', $i, $elementName, $expectedMimeTypes[$i]));
        $this->assertEquals(0, $file->getSize(), sprintf('The file for upload #%d of element "%s" has a size of 0 bytes.', $i, $elementName));
        $actualScheme = parse_url($file->getFileUri(), PHP_URL_SCHEME);
        $this->assertEquals($expectedScheme, $actualScheme, sprintf('The file for upload #%d of element "%s" is saved in the %s filesystem.', $i, $elementName, $expectedScheme));
      }
    }
  }

  /**
   * Tests uploading files that do not meet the validation criteria.
   */
  public function testFileUploadWithValidationErrors(): void {
    // Create some test files to upload.
    foreach (['txt', 'mp3', 'flac'] as $extension) {
      $this->files[$extension] = $this->fileSystem->getTempDirectory() . '/graphql_webform_upload_test.' . $extension;
      touch($this->files[$extension]);
      // Create a file with a size of 1.000001MB. This is intended to trigger a
      // validation error for the file_upload element which is limited to 1MB.
      if ($extension === 'txt') {
        file_put_contents($this->files[$extension], str_repeat('a', 1024 * 1024 + 1));
      }
    }

    $request = Request::create(
      uri: '/graphql/test',
      method: 'POST',
      parameters: [
        'query' => $this->getQueryFromFile('submission_with_file_upload.gql'),
        'variables' => [
          'elements' => [
            (object) [
              'element' => 'required_text_field',
              'value' => 'A value.',
            ],
          ],
          'files' => [
            ['element' => 'file_upload', 'file' => NULL],
            ['element' => 'audio_files', 'file' => NULL],
            ['element' => 'audio_files', 'file' => NULL],
          ],
          'id' => 'graphql_webform_test_form',
        ],
        'map' => [
          '0' => ['variables.files.0.file'],
          '1' => ['variables.files.1.file'],
          '2' => ['variables.files.2.file'],
        ],
      ],
      files: [
        '0' => [
          'name' => 'test.txt',
          'type' => 'text/plain',
          'size' => 1024 * 1024 + 1,
          'tmp_name' => $this->files['txt'],
          'error' => UPLOAD_ERR_OK,
        ],
        '1' => [
          'name' => 'audio_file.mp3',
          'type' => 'audio/mpeg',
          'size' => 0,
          'tmp_name' => $this->files['mp3'],
          'error' => UPLOAD_ERR_OK,
        ],
        '2' => [
          'name' => 'audio_file.flac',
          'type' => 'audio/flac',
          'size' => 0,
          'tmp_name' => $this->files['flac'],
          'error' => UPLOAD_ERR_OK,
        ],
      ],
    );

    $result = $this->executeMultiPartRequest($request);

    $this->assertNotEmpty($result->data->submitWebform ?? [], 'The response contains information about the submission.');
    $resultData = $result->data->submitWebform;

    $this->assertEmpty($resultData->errors, 'There are no errors.');
    $expectedValidationErrors = [
      'file_upload' => [
        'The file is <em class="placeholder">1 MB</em> exceeding the maximum file size of <em class="placeholder">1 MB</em>.',
      ],
      'audio_files' => [
        'Only files with the following extensions are allowed: <em class="placeholder">mp3 ogg wav</em>.',
      ],
    ];
    $this->assertValidationErrors($result, $expectedValidationErrors);

    $this->assertEmpty($resultData->submission, 'No submission was returned.');
    $this->assertEmpty(WebformSubmission::loadMultiple(), 'No submission entity was created.');
  }

  /**
   * Tests uploading too many files.
   *
   * The 'file_upload' field only accepts a single file, and the 'audio_files'
   * field accepts up to two files. We will attempt to upload more.
   */
  public function testUploadTooManyFiles(): void {
    // Create some test files to upload.
    foreach (['txt', 'jpg', 'mp3', 'ogg', 'wav'] as $extension) {
      $this->files[$extension] = $this->fileSystem->getTempDirectory() . '/graphql_webform_upload_test.' . $extension;
      touch($this->files[$extension]);
    }

    $request = Request::create(
      uri: '/graphql/test',
      method: 'POST',
      parameters: [
        'query' => $this->getQueryFromFile('submission_with_file_upload.gql'),
        'variables' => [
          'elements' => [
            (object) [
              'element' => 'required_text_field',
              'value' => 'A value.',
            ],
          ],
          'files' => [
            ['element' => 'file_upload', 'file' => NULL],
            ['element' => 'file_upload', 'file' => NULL],
            ['element' => 'audio_files', 'file' => NULL],
            ['element' => 'audio_files', 'file' => NULL],
            ['element' => 'audio_files', 'file' => NULL],
          ],
          'id' => 'graphql_webform_test_form',
        ],
        'map' => [
          '0' => ['variables.files.0.file'],
          '1' => ['variables.files.1.file'],
          '2' => ['variables.files.2.file'],
          '3' => ['variables.files.3.file'],
          '4' => ['variables.files.4.file'],
        ],
      ],
      files: [
        '0' => [
          'name' => 'test.txt',
          'type' => 'text/plain',
          'size' => 0,
          'tmp_name' => $this->files['txt'],
          'error' => UPLOAD_ERR_OK,
        ],
        '1' => [
          'name' => 'test.jpg',
          'type' => 'image/jpeg',
          'size' => 0,
          'tmp_name' => $this->files['jpg'],
          'error' => UPLOAD_ERR_OK,
        ],
        '2' => [
          'name' => 'audio_file.mp3',
          'type' => 'audio/mpeg',
          'size' => 0,
          'tmp_name' => $this->files['mp3'],
          'error' => UPLOAD_ERR_OK,
        ],
        '3' => [
          'name' => 'audio_file.ogg',
          'type' => 'audio/ogg',
          'size' => 0,
          'tmp_name' => $this->files['ogg'],
          'error' => UPLOAD_ERR_OK,
        ],
        '4' => [
          'name' => 'audio_file.wav',
          'type' => 'audio/wav',
          'size' => 0,
          'tmp_name' => $this->files['wav'],
          'error' => UPLOAD_ERR_OK,
        ],
      ],
    );

    $result = $this->executeMultiPartRequest($request);

    $this->assertNotEmpty($result->data->submitWebform ?? [], 'The response contains information about the submission.');
    $resultData = $result->data->submitWebform;

    $this->assertEmpty($resultData->errors, 'There are no errors.');

    $expectedValidationErrors = [
      'file_upload' => [
        'Only one file can be uploaded.',
      ],
      'audio_files' => [
        'The number of files uploaded exceeds the maximum of 2.',
      ],
    ];
    $this->assertValidationErrors($result, $expectedValidationErrors);

    $this->assertEmpty($resultData->submission, 'No submission was returned.');
    $this->assertEmpty(WebformSubmission::loadMultiple(), 'No submission entity was created.');
    $this->assertEmpty($this->fileStorage->loadMultiple(), 'No file entities were created.');
  }

  /**
   * Executes a request and returns the response.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request to execute.
   *
   * @return object
   *   The response object.
   */
  protected function executeMultiPartRequest(Request $request): object {
    $request->headers->add(['content-type' => 'multipart/form-data']);
    $response = $this->container->get('http_kernel')->handle($request);
    return json_decode($response->getContent());
  }

  /**
   * Asserts that the validation errors in the response are as expected.
   *
   * @param object $result
   *   The response object.
   * @param array $expectedValidationErrors
   *   An array of expected validation errors. The keys are the element names,
   *   and the values are arrays of expected validation messages.
   */
  protected function assertValidationErrors(object $result, array $expectedValidationErrors): void {
    $this->assertNotEmpty($result->data->submitWebform ?? [], 'The response contains information about the submission.');
    $resultData = $result->data->submitWebform;
    $this->assertIsArray($resultData->validationErrors, 'The response contains validation errors.');

    foreach ($expectedValidationErrors as $elementName => $expectedMessages) {
      $elementErrors = array_filter($resultData->validationErrors, static fn ($error): bool => $error->element === $elementName);
      $this->assertCount(1, $elementErrors, sprintf('The validation errors for the "%s" element are grouped together.', $elementName));
      $elementError = reset($elementErrors);
      $this->assertEquals($elementName, $elementError->element, sprintf('The validation error for the "%s" element has the correct element ID.', $elementName));
      $this->assertCount(count($expectedMessages), $elementError->messages, sprintf('There are %d validation messages for the "%s" element.', count($expectedMessages), $elementName));
      foreach ($expectedMessages as $i => $expectedMessage) {
        $this->assertEquals($expectedMessage, $elementError->messages[$i], sprintf('Validation message #%d for the "%s" element is correct.', $i, $elementName));
      }
    }
  }

}

namespace Symfony\Component\HttpFoundation\File;

/**
 * Mock the PHP function is_uploaded_file().
 *
 * Since we are not *really* uploading a file through the webserver, PHP will
 * not recognize the file as an uploaded file. We mock the function to return
 * TRUE for our test files.
 *
 * @param string $filename
 *   The filename being checked.
 *
 * @return bool
 *   Will return TRUE for our test files.
 */
function is_uploaded_file($filename) {
  $temp_dir = \Drupal::service('file_system')->getTempDirectory();
  $prefix = $temp_dir . '/graphql_webform_upload_test.';
  return str_starts_with($filename, $prefix);
}
