<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel\Query;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Tests\graphql_webform\Kernel\GraphQLWebformKernelTestBase;
use Drupal\graphql_webform\Enum\WebformSubmissionConfirmationType;
use Drupal\webform\Entity\WebformSubmission;
use Drupal\webform\WebformSubmissionInterface;

/**
 * Tests querying a submission confirmation.
 *
 * @group graphql_webform
 */
class WebformConfirmationTest extends GraphQLWebformKernelTestBase {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('webform_submission');

    $this->entityTypeManager = $this->container->get('entity_type.manager');
  }

  /**
   * Tests passing invalid arguments to the webform confirmation query.
   *
   * @dataProvider invalidArgumentsProvider
   */
  public function testInvalidArguments(array $arguments): void {
    $query = $this->getQueryFromFile('webform_confirmation.gql');

    // Error responses are not cached.
    $cacheMetadata = $this->defaultCacheMetaData()
      ->setCacheContexts(['user.permissions'])
      ->setCacheTags($this->getGraphQlCacheTags())
      ->setCacheMaxAge(0);

    $this->assertErrors($query, $arguments, [
      'Either a webform or a webform submission must be provided.',
    ], $cacheMetadata);
  }

  /**
   * Provides data for the testInvalidArguments test.
   *
   * @return array
   *   An array of test cases.
   */
  public static function invalidArgumentsProvider(): array {
    return [
      'no arguments' => [[]],
      'only token' => [['token' => 'some-token']],
    ];
  }

  /**
   * Tests querying a submission confirmation with an invalid token.
   */
  public function testInvalidToken(): void {
    $query = $this->getQueryFromFile('webform_confirmation.gql');

    // Error responses are not cached.
    $cacheMetadata = $this->defaultCacheMetaData()
      ->setCacheContexts(['user.permissions'])
      ->setCacheTags($this->getGraphQlCacheTags())
      ->setCacheMaxAge(0);

    $this->assertErrors($query, [
      'submissionId' => (int) $this->createWebformSubmission()->id(),
      'token' => 'invalid-token',
    ], [
      'Invalid webform submission token.',
    ], $cacheMetadata);
  }

  /**
   * Tests querying a submission confirmation while passing a webform ID.
   *
   * This is typically used for displaying a confirmation message for webforms
   * that do not save submissions.
   */
  public function testWithWebformId(): void {
    $this->config('webform.webform.graphql_webform_test_form')
      // Disable the saving of submissions.
      ->set('settings.results_disabled', TRUE)
      // Set a custom confirmation title that is suitable for a submissionless
      // form.
      ->set('settings.confirmation_title', 'Thanks for filling in [webform:title].')
      ->save();

    // Clear the static cache to ensure the updated configuration is used.
    $this->entityTypeManager->getStorage('webform')->resetCache();

    $query = $this->getQueryFromFile('webform_confirmation.gql');

    $this->assertResults($query, [
      'webformId' => 'graphql_webform_test_form',
    ], [
      'webformConfirmation' => [
        'message' => 'New submission added to GraphQL Webform test form.',
        'type' => WebformSubmissionConfirmationType::UrlMessage->toGraphQlEnumValue(),
        'title' => 'Thanks for filling in GraphQL Webform test form.',
        'redirectUrl' => 'form/submit',
      ],
    ], $this->defaultCacheMetaData());
  }

  /**
   * Tests querying a submission confirmation while passing a submission ID.
   */
  public function testWithSubmissionId(): void {
    $submission = $this->createWebformSubmission();

    $query = $this->getQueryFromFile('webform_confirmation.gql');

    $cacheMetadata = $this->defaultCacheMetaData()
      ->setCacheContexts(['user.permissions'])
      ->setCacheTags(array_merge($this->getGraphQlCacheTags(), ['webform_submission:' . $submission->id()]));

    $this->assertResults($query, [
      'submissionId' => (int) $submission->id(),
      'token' => $submission->getToken(),
    ], [
      'webformConfirmation' => [
        'message' => 'New submission added to GraphQL Webform test form.',
        'type' => WebformSubmissionConfirmationType::UrlMessage->toGraphQlEnumValue(),
        'title' => 'Your submission has been Completed',
        'redirectUrl' => 'form/submit',
      ],
    ], $cacheMetadata);
  }

  /**
   * Creates a test webform submission.
   *
   * @return \Drupal\webform\WebformSubmissionInterface
   *   The created webform submission.
   */
  protected function createWebformSubmission(): WebformSubmissionInterface {
    /** @var \Drupal\webform\WebformSubmissionInterface $submission */
    $submission = WebformSubmission::create([
      'webform_id' => 'graphql_webform_test_form',
    ]);
    $submission->save();
    return $submission;
  }

  /**
   * Returns the GraphQL specific cache tags for a response.
   *
   * @return string[]
   *   The cache tags.
   */
  protected function getGraphQlCacheTags(): array {
    return ['graphql_response', 'config:graphql.graphql_servers.' . $this->server->id()];
  }

}
