<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Unit\Model;

use Drupal\Tests\UnitTestCase;
use Drupal\graphql_webform\Model\WebformSubmissionResult;
use Drupal\graphql_webform\Model\WebformSubmissionValidationError;
use Drupal\webform\WebformSubmissionInterface;

/**
 * @coversDefaultClass \Drupal\graphql_webform\Model\WebformSubmissionResult
 * @group graphql_webform
 */
class WebformSubmissionResultTest extends UnitTestCase {

  /**
   * @covers ::setSubmission
   * @covers ::getSubmission
   */
  public function testSubmission() {
    $submission = $this->createMock(WebformSubmissionInterface::class);
    $result = new WebformSubmissionResult();
    $result->setSubmission($submission);
    $this->assertSame($submission, $result->getSubmission());
  }

  /**
   * @covers ::addError
   * @covers ::getErrors
   */
  public function testAddError() {
    $result = new WebformSubmissionResult();
    $result->addError('Test error');
    $this->assertEquals(['Test error'], $result->getErrors());
  }

  /**
   * @covers ::addValidationError
   * @covers ::getOrCreateValidationError
   * @covers ::getValidationErrors
   * @covers ::hasValidationError
   */
  public function testAddValidationError() {
    $result = new WebformSubmissionResult();
    $this->assertEmpty($result->getValidationErrors(), 'A new result does not have validation errors.');
    $this->assertFalse($result->hasValidationError('element_id'), 'A new result does not have validation errors for an element.');
    $this->assertFalse($result->hasValidationError(NULL), 'A new result does not have a general validation error.');

    // Add a first validation error.
    $result->addValidationError('Validation error', 'element_id');
    $errors = $result->getValidationErrors();
    $this->assertCount(1, $errors);
    $returnedError = reset($errors);
    $this->assertEquals(['Validation error'], $returnedError->getMessages());
    $this->assertEquals('element_id', $returnedError->getElementId());
    $this->assertTrue($result->hasValidationError('element_id'), 'The result has a validation error for the element.');
    $this->assertFalse($result->hasValidationError('another_element_id'), 'The result does not have a validation error for another element.');
    $this->assertFalse($result->hasValidationError(NULL), 'The result does not have a general validation error.');

    // Add another validation error for the same element.
    $result->addValidationError('Another validation error', 'element_id');
    $errors = $result->getValidationErrors();
    $this->assertCount(1, $errors);
    $returnedError = reset($errors);
    $this->assertEquals(['Validation error', 'Another validation error'], $returnedError->getMessages());
    $this->assertEquals('element_id', $returnedError->getElementId());
    $this->assertTrue($result->hasValidationError('element_id'), 'The result has a validation error for the element.');
    $this->assertFalse($result->hasValidationError('another_element_id'), 'The result does not have a validation error for another element.');
    $this->assertFalse($result->hasValidationError(NULL), 'The result does not have a general validation error.');

    // Add a validation error for a different element.
    $result->addValidationError('Yet another validation error', 'another_element_id');
    $errors = $result->getValidationErrors();
    $this->assertCount(2, $errors);
    $error_element_1 = $result->getOrCreateValidationError('element_id');
    $this->assertEquals(['Validation error', 'Another validation error'], $error_element_1->getMessages());
    $this->assertEquals('element_id', $error_element_1->getElementId());
    $error_element_2 = $result->getOrCreateValidationError('another_element_id');
    $this->assertEquals(['Yet another validation error'], $error_element_2->getMessages());
    $this->assertEquals('another_element_id', $error_element_2->getElementId());
    $this->assertTrue($result->hasValidationError('element_id'), 'The result has a validation error for the first element.');
    $this->assertTrue($result->hasValidationError('another_element_id'), 'The result has a validation error for the second element.');
    $this->assertFalse($result->hasValidationError(NULL), 'The result does not have a general validation error.');

    // Add a validation error without an element ID.
    $result->addValidationError('Validation error without element ID', NULL);
    $errors = $result->getValidationErrors();
    $this->assertCount(3, $errors);
    $error_no_element = $result->getOrCreateValidationError(NULL);
    $this->assertEquals(['Validation error without element ID'], $error_no_element->getMessages());
    $this->assertNull($error_no_element->getElementId());
    $this->assertTrue($result->hasValidationError('element_id'), 'The result has a validation error for the first element.');
    $this->assertTrue($result->hasValidationError('another_element_id'), 'The result has a validation error for the second element.');
    $this->assertTrue($result->hasValidationError(NULL), 'The result has a general validation error.');
  }

  /**
   * @covers ::normalizeElementId
   * @covers ::addValidationError
   * @covers ::getOrCreateValidationError
   * @covers ::hasValidationError
   */
  public function normalizedElementIdTest(): void {
    // Try to add and retrieve validation errors using an element ID which is
    // derived from a nested form element. The Form API uses a different format
    // for nested elements than what we want to present to the GraphQL client.
    $originalElementId = 'parent][child][grandchild';
    $expectedElementId = 'parent[child][grandchild]';
    $message = 'Validation error for nested element';

    // Add a validation error using the original element ID. It should be stored
    // using the normalized element ID.
    $result = new WebformSubmissionResult();
    $result->addValidationError($message, $originalElementId);
    $errors = $result->getValidationErrors();
    $this->assertCount(1, $errors);
    $error = reset($errors);
    $this->assertEquals($expectedElementId, $error->getElementId(), 'The element ID was normalized when a validation error was added.');
    $this->assertEquals([$message], $error->getMessages(), 'The error message was added.');

    // Try to create an empty validation error using the original element ID.
    $result = new WebformSubmissionResult();
    $error = $result->getOrCreateValidationError($originalElementId);
    $this->assertInstanceOf(WebformSubmissionValidationError::class, $error, 'The correct error object is returned.');
    $this->assertEquals($expectedElementId, $error->getElementId(), 'The element ID was normalized when a validation error was created.');
    $this->assertEmpty($error->getMessages());

    // Try to check for the existence of a validation error using the original
    // and normalized element IDs. Both should work.
    $this->assertTrue($result->hasValidationError($originalElementId), 'The element ID was normalized when checking for a validation error.');
    $this->assertTrue($result->hasValidationError($expectedElementId), 'It is possible to check for the existence of a validation error using the normalized element ID.');
  }

  /**
   * @covers ::isValid
   */
  public function testIsValid() {
    $result = new WebformSubmissionResult();
    $this->assertTrue($result->isValid());

    $result->addError('Test error');
    $this->assertFalse($result->isValid());

    $result = new WebformSubmissionResult();
    $result->addValidationError('Validation error', 'element_id');
    $this->assertFalse($result->isValid());

    $result = new WebformSubmissionResult();
    $result->addValidationError('Validation error', NULL);
    $this->assertFalse($result->isValid());
  }

}
