<?php

declare(strict_types=1);

namespace Drupal\group_domain\Plugin\views\filter;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeManagerInterface;
use Drupal\group_domain\GroupDomainInfo;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Drupal\views\Views;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * If active, views will show only entities from the current domain group.
 *
 * @ingroup views_filter_handlers
 *
 * @ViewsFilter("domain_group_current_group")
 */
class DomainGroupCurrentGroup extends FilterPluginBase {

  /**
   * Group Domain Info service.
   */
  private GroupDomainInfo $groupDomainInfo;

  /**
   * Entity Type Manager.
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Group Relation Type Manager.
   */
  private GroupRelationTypeManagerInterface $groupRelationTypeManager;

  /**
   * The constructor.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    GroupDomainInfo $group_domain_info,
    EntityTypeManagerInterface $entity_type_manager,
    GroupRelationTypeManagerInterface $group_relation_type_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->groupDomainInfo = $group_domain_info;
    $this->entityTypeManager = $entity_type_manager;
    $this->groupRelationTypeManager = $group_relation_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('group_domain.info'),
      $container->get('entity_type.manager'),
      $container->get('group_relation_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    $contexts = parent::getCacheContexts();
    $contexts[] = 'group_domain';
    return $contexts;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    // We need to invalidate every time domain group type relationships
    // are modified. Possibly could be optimized a bit if we knew which types
    // of relationships does a specific view rely on.
    $tags = [];
    $group_domain_settings = $this->entityTypeManager->getStorage('group_domain_settings')->loadMultiple();
    /** @var \Drupal\group\Entity\Storage\GroupRelationshipTypeStorage */
    $group_relationship_type_storage = $this->entityTypeManager->getStorage('group_relationship_type');
    $plugin_map = $this->groupRelationTypeManager->getGroupTypePluginMap();
    foreach ($group_domain_settings as $setting) {
      $group_type_id = $setting->id();
      if (!\array_key_exists($group_type_id, $plugin_map)) {
        continue;
      }
      foreach ($plugin_map[$group_type_id] as $plugin_id) {
        $tags[] = 'group_relationship_list:' . $group_relationship_type_storage->getRelationshipTypeId($group_type_id, $plugin_id);
      }
    }
    return $tags;
  }

  /**
   * Add this filter to the query.
   */
  public function query(): void {
    $group = $this->groupDomainInfo->getCurrentDomainGroup();

    // Don't filter if not on a group domain.
    if ($group === NULL) {
      return;
    }

    $entity_type_id = $this->getEntityType();
    $table_alias = $this->ensureMyTable();
    $group_relationship = $this->entityTypeManager->getDefinition('group_relationship');
    $group_relationship_data_table = $group_relationship->getDataTable();

    if ($entity_type_id === 'group_relationship') {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $join = Views::pluginManager('join')->createInstance('standard', [
        'table' => $group_relationship_data_table,
        'field' => $entity_type->getKey('id'),
        'left_table' => $table_alias,
        'left_field' => $entity_type->getKey('id'),
        'type' => 'INNER',
        'operator' => '=',
      ]);
      $group_relationship_data_table_alias = $this->query->addRelationship($group_relationship_data_table, $join, $table_alias);
      $this->query->addWhere($this->options['group'], $group_relationship_data_table_alias . '.gid', $group->id(), '=');
      return;
    }

    $plugin_ids = $this->groupRelationTypeManager->getPluginIdsByEntityTypeId($entity_type_id);

    $group_relationship_type_ids = [];
    foreach ($plugin_ids as $plugin_id) {
      $group_relationship_type_ids = \array_unique(\array_merge(
        $group_relationship_type_ids,
        $this->groupRelationTypeManager->getRelationshipTypeIds($plugin_id),
      ));
    }
    if (\count($group_relationship_type_ids) === 0) {
      return;
    }

    $entity_type = $this->entityTypeManager->getDefinition($this->getEntityType());

    $configuration = [
      'table' => $group_relationship_data_table,
      'field' => 'entity_id',
      'left_table' => $table_alias,
      'left_field' => $entity_type->getKey('id'),
      'type' => 'INNER',
      'operator' => '=',
    ];

    $join = Views::pluginManager('join')->createInstance('standard', $configuration);
    $group_relationship_data_table_alias = $this->query->addRelationship($group_relationship_data_table, $join, $table_alias);

    $this->query->addWhere($this->options['group'], $group_relationship_data_table_alias . '.' . $group_relationship->getKey('bundle'), $group_relationship_type_ids, 'IN');
    $this->query->addWhere($this->options['group'], $group_relationship_data_table_alias . '.gid', $group->id(), '=');
  }

}
