<?php

namespace Drupal\group_inline_entity_form\Plugin\Field\FieldWidget;

use Drupal\inline_entity_form\Plugin\Field\FieldWidget\InlineEntityFormComplex;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\Context\ContextProviderInterface;
use Drupal\group\Access\GroupPermissionCheckerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Field\Attribute\FieldWidget;

/**
 * Complex inline widget with group module support.
 */
#[FieldWidget(
  id: 'group_inline_entity_form_complex',
  label: new TranslatableMarkup('Inline entity form - Complex - Group'),
  field_types: [
    'entity_reference',
    'entity_reference_revisions',
  ],
  multiple_values: TRUE,
)]
class GroupInlineEntityFormComplex extends InlineEntityFormComplex {

  /**
   * The Group relation type manager.
   *
   * @var \Drupal\group\Plugin\Group\Relation\GroupRelationTypeManager
   */
  protected $groupRelationTypeManager;

  /**
   * The group permission access checker service.
   *
   * @var \Drupal\group\Access\GroupPermissionCheckerInterface
   */
  protected $groupPermissionChecker;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $account;

  /**
   * The group context from the route.
   *
   * @var \Drupal\Core\Plugin\Context\ContextInterface
   */
  protected $context;

  /**
   * The group entity from the route.
   *
   * @var \Drupal\group\Entity\GroupInterface
   */
  protected $group;

  /**
   * Constructs a InlineEntityFormComplex object.
   *
   * @param string $plugin_id
   *   The plugin_id for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   Module handler service.
   * @param \Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface $selection_manager
   *   The selection plugin manager.
   * @param \Drupal\group\Plugin\Group\Relation\GroupRelationTypeManagerInterface $group_relation_type_manager
   *   The Group relation type manager.
   * @param \Drupal\group\Access\GroupPermissionCheckerInterface $group_permission_checker
   *   The group permission checker.
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The current user.
   * @param \Drupal\Core\Plugin\Context\ContextProviderInterface $context_provider
   *   The group context provider.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    EntityTypeManagerInterface $entity_type_manager,
    EntityDisplayRepositoryInterface $entity_display_repository,
    ModuleHandlerInterface $module_handler,
    SelectionPluginManagerInterface $selection_manager,
    GroupRelationTypeManagerInterface $group_relation_type_manager,
    GroupPermissionCheckerInterface $group_permission_checker,
    AccountProxyInterface $account,
    ContextProviderInterface $context_provider,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings, $entity_type_bundle_info, $entity_type_manager, $entity_display_repository, $module_handler, $selection_manager);
    $this->groupRelationTypeManager = $group_relation_type_manager;
    $this->groupPermissionChecker = $group_permission_checker;
    $this->account = $account;

    $contexts = $context_provider->getRuntimeContexts(['group']);
    $this->context = $contexts['group'];
    $this->group = $this->context->getContextValue();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_type.bundle.info'),
      $container->get('entity_type.manager'),
      $container->get('entity_display.repository'),
      $container->get('module_handler'),
      $container->get('plugin.manager.entity_reference_selection'),
      $container->get('group_relation_type.manager'),
      $container->get('group_permission.checker'),
      $container->get('current_user'),
      $container->get('group.group_route_context')
    );
  }

  /**
   * Get the relationship plugin id for the given entity type + bundle.
   *
   * @param string $entity_type_id
   *   An entity type id.
   * @param string $bundle
   *   An entity bundle.
   *
   * @return string|bool
   *   The plugin id.
   */
  protected function getRelationshipPluginId($entity_type_id, $bundle) {
    foreach ($this->groupRelationTypeManager->getDefinitions() as $plugin_id => $plugin_info) {
      if ($plugin_info->getEntityTypeId() === $entity_type_id) {
        if (!empty($plugin_info->getEntityBundle()) && $plugin_info->getEntityBundle() === $bundle) {
          return $plugin_id;
        }
      }
    }
    return FALSE;
  }

  /**
   * Gets the bundles for which the current user has create access.
   *
   * @return string[]
   *   The list of bundles.
   */
  protected function getCreateBundles() {
    // If no group is found in context, fallback to core formatter.
    if (empty($this->group)) {
      return parent::getCreateBundles();
    }

    $op = 'create';
    $target_type = $this->getFieldSetting('target_type');

    $create_bundles = [];
    foreach ($this->getTargetBundles() as $bundle) {
      $plugin_id = $this->getRelationshipPluginId($target_type, $bundle);
      $permission = "{$op} {$plugin_id} entity";
      if ($this->groupPermissionChecker->hasPermissionInGroup($permission, $this->account, $this->group)) {
        $create_bundles[] = $bundle;
      }
    }

    return $create_bundles;
  }

}
