<?php

namespace Drupal\group_inline_entity_form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\group\Entity\GroupRelationshipTypeInterface;
use Drupal\group\Entity\GroupInterface;

/**
 * Performs widget submission.
 *
 * Widgets don't save changed entities, nor do they delete removed entities.
 * Instead, they flag them so that changes are only applied when the main form
 * is submitted.
 */
class WidgetSubmit {

  /**
   * Attaches the widget submit functionality to the given form.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function attach(array &$form, FormStateInterface $form_state) {
    // Register preSubmit function at the beginning.
    array_unshift($form['#ief_element_submit'], [get_called_class(), 'preSubmit']);
    // Register postSubmit function after IEF's doSubmit.
    $form['#ief_element_submit'][] = [get_called_class(), 'postSubmit'];
  }

  /**
   * Marks new entities for later.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function preSubmit(array $form, FormStateInterface $form_state) {
    $widget_states =& $form_state->get('inline_entity_form');
    // Widget states can be in an arbitrary order depending on user's form
    // interaction. We sort them lexicographically in reverse order to get inner
    // forms before outer forms, to ensure inside-out saving of entities.
    // @see \Drupal\inline_entity_form\Plugin\Field\FieldWidget\InlineEntityFormBase::makeIefId
    $widget_states = $widget_states ?? [];
    krsort($widget_states, SORT_STRING);
    foreach ($widget_states as &$widget_state) {
      $widget_state += ['entities' => [], 'delete' => []];
      // dpm($widget_state['entities']);.
      foreach ($widget_state['entities'] as &$entity_item) {
        if (!empty($entity_item['entity']) && !empty($entity_item['needs_save'])) {
          if ($entity_item['entity']->isNew()) {
            $entity_item['is_new'] = TRUE;
          }
        }
      }
    }
  }

  /**
   * Attach all marked entities to the current group.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function postSubmit(array $form, FormStateInterface $form_state) {
    $widget_states =& $form_state->get('inline_entity_form');
    // Widget states can be in an arbitrary order depending on user's form
    // interaction. We sort them lexicographically in reverse order to get inner
    // forms before outer forms, to ensure inside-out saving of entities.
    // @see \Drupal\inline_entity_form\Plugin\Field\FieldWidget\InlineEntityFormBase::makeIefId
    $widget_states = $widget_states ?? [];
    krsort($widget_states, SORT_STRING);

    $contexts = \Drupal::service('group.group_route_context')->getRuntimeContexts(['group']);
    $group = $contexts['group']->getContextValue();

    // No group was found, exit...
    if (!($group instanceof GroupInterface)) {
      return;
    }

    $registry = static::getContentPluginDefinitions();

    foreach ($widget_states as &$widget_state) {
      $widget_state += ['entities' => [], 'delete' => []];
      foreach ($widget_state['entities'] as &$entity_item) {
        if (!empty($entity_item['entity']) && !empty($entity_item['is_new'])) {
          if (!empty($registry[$group->bundle()][$entity_item['entity']->getEntityTypeId()][$entity_item['entity']->bundle()])) {
            $group->addRelationship(
              $entity_item['entity'],
              $registry[$group->bundle()][$entity_item['entity']->getEntityTypeId()][$entity_item['entity']->bundle()]
            );
          }
        }
      }
    }
  }

  /**
   * Get an array of available group contents.
   *
   * @return array
   *   Array of of available group contents.
   */
  public static function getContentPluginDefinitions() {
    $storage = \Drupal::entityTypeManager()->getStorage('group_relationship_type');
    $group_relationship_types = $storage->loadMultiple();

    $registry = [];
    foreach ($group_relationship_types as $group_relationship_type) {
      assert($group_relationship_type instanceof GroupRelationshipTypeInterface);
      $group_type_id = $group_relationship_type->getGroupTypeId();
      $entity_type_id = $group_relationship_type->getPlugin()->getPluginDefinition()->getEntityTypeId();
      $entity_bundle = $group_relationship_type->getPlugin()->getPluginDefinition()->getEntityBundle();
      $registry[$group_type_id][$entity_type_id][$entity_bundle] = $group_relationship_type->getPluginId();
    }
    return $registry;
  }

}
