<?php

namespace Drupal\group_jsonapi_create_access\Access;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\group\Entity\GroupInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Route;

/**
 * Defines an access checker for group_relationship creation via JSON:API.
 */
class GroupJsonapiCreateAccessCheck implements AccessInterface {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The key used by the routing requirement.
   *
   * @var string
   */
  protected $requirementsKey = '_group_jsonapi_create_access';

  /**
   * Constructs a GroupJsonapiCreateAccessCheck object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, RequestStack $request_stack) {
    $this->entityTypeManager = $entity_type_manager;
    $this->requestStack = $request_stack;
  }

  /**
   * Checks access to create the group_relationship entity.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The parametrized route.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The currently logged in account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route, RouteMatchInterface $route_match, AccountInterface $account) {
    [$entity_type, $bundle] = explode(':', $route->getRequirement($this->requirementsKey) . ':');

    // Extract group from JSON:API request body.
    $context = [];
    $request = $this->requestStack->getCurrentRequest();
    if ($request && $request->getRequestFormat() === 'api_json') {
      $content = $request->getContent();
      if (!empty($content)) {
        try {
          $body = Json::decode($content);
          if (isset($body['data']['relationships']['gid']['data']['id'])) {
            $group_uuid = $body['data']['relationships']['gid']['data']['id'];
            if (!empty($group_uuid)) {
              $groups = $this->entityTypeManager
                ->getStorage('group')
                ->loadByProperties(['uuid' => $group_uuid]);
              if (!empty($groups)) {
                $context['group'] = reset($groups);
              }
            }
          }
        }
        catch (\JsonException $e) {
          // If JSON decode fails, continue without group context.
        }
      }
    }

    // If we couldn't find the group, deny access.
    if (!isset($context['group']) || !($context['group'] instanceof GroupInterface)) {
      return AccessResult::forbidden('Group not available from request. Group may not exist or jsonapi request body may be malformed.');
    }

    // Call the original access check with the group in context.
    return $this->entityTypeManager->getAccessControlHandler($entity_type)->createAccess($bundle, $account, $context, TRUE);
  }

}
