<?php

namespace Drupal\Tests\group_jsonapi_create_access\Functional;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Url;
use Drupal\group\Entity\Group;
use Drupal\group\Entity\GroupRelationshipType;
use Drupal\group\Entity\GroupType;
use Drupal\Tests\jsonapi\Functional\JsonApiFunctionalTestBase;

/**
 * Tests JSON:API create access for group_relationship entities.
 *
 * @group group_jsonapi_create_access
 */
class GroupRelationshipJsonapiCreateAccessTest extends JsonApiFunctionalTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'group_jsonapi_create_access',
    'group',
    'jsonapi',
    'basic_auth',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A test group.
   *
   * @var \Drupal\group\Entity\GroupInterface
   */
  protected $group;

  /**
   * The group type ID.
   *
   * @var string
   */
  protected $groupTypeId = 'test_group_type';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Enable JSON:API write operations.
    \Drupal::configFactory()->getEditable('jsonapi.settings')
      ->set('read_only', FALSE)
      ->save();

    // Create a test group type.
    $group_type = GroupType::load($this->groupTypeId);
    if (!$group_type) {
      $group_type = GroupType::create([
        'id' => $this->groupTypeId,
        'label' => 'Test Group Type',
      ]);
      $group_type->save();

      // Rebuild router to ensure our route subscriber runs
      // after group type creation.
      \Drupal::service('router.builder')->rebuild();
    }

    // Create a group.
    $this->group = Group::create([
      'type' => $this->groupTypeId,
      'label' => 'Test Group',
    ]);
    $this->group->save();

    // Load or create the admin role for this group type.
    $admin_role_id = $this->groupTypeId . '-admin';
    $role_storage = \Drupal::entityTypeManager()->getStorage('group_role');
    $admin_role = $role_storage->load($admin_role_id);
    if (!$admin_role) {
      // Create the admin role if it doesn't exist.
      $admin_role = $role_storage->create([
        'id' => $admin_role_id,
        'label' => 'Administrator',
        'weight' => -10,
        'internal' => FALSE,
        'scope' => 'individual',
        'group_type' => $this->groupTypeId,
        'permissions' => [
          'view group',
          'administer members',
          'create group_membership relationship',
        ],
      ]);
      $admin_role->save();
    }

  }

  /**
   * Tests that the route is properly modified to use our custom access check.
   */
  public function testRouteIsModified(): void {
    $relationship_type = GroupRelationshipType::load('test_group_type-group_membership');
    $route_name = 'jsonapi.group_relationship--' . $relationship_type->id() . '.collection.post';
    $route_provider = \Drupal::service('router.route_provider');
    $route = $route_provider->getRouteByName($route_name);
    $requirements = $route->getRequirements();

    $this->assertArrayHasKey('_group_jsonapi_create_access', $requirements, 'Route should have _group_jsonapi_create_access requirement');
    $this->assertArrayNotHasKey('_entity_create_access', $requirements, 'Route should NOT have _entity_create_access requirement');
    $this->assertEquals('group_relationship:test_group_type-group_membership', $requirements['_group_jsonapi_create_access']);
  }

  /**
   * Tests user with permission can create relationship via JSON:API.
   */
  public function testAllowedCreateAccess(): void {
    // Create user & verify access.
    $allowed_user = $this->drupalCreateUser(['access content']);
    $allowed_user->save();
    $this->group->addMember($allowed_user, ['group_roles' => [$this->groupTypeId . '-admin']]);
    $this->assertNotEmpty($this->group->getMember($allowed_user));
    $this->assertTrue($this->group->access('view', $allowed_user), 'User should be able to view the group');
    $handler = \Drupal::entityTypeManager()->getAccessControlHandler('group_relationship');
    $this->drupalLogin($allowed_user);
    $this->assertTrue($handler->createAccess($this->groupTypeId . "-group_membership", $allowed_user, ['group' => $this->group]));
    $this->drupalLogout();

    // Build the JSON:API request body.
    $relationship_type = GroupRelationshipType::load($this->groupTypeId . '-group_membership');
    $resource_type_name = 'group_relationship--' . $relationship_type->id();
    $user_to_add = $this->drupalCreateUser();
    $doc = [
      'data' => [
        'type' => $resource_type_name,
        'relationships' => [
          'gid' => [
            'data' => [
              'type' => 'group--' . $this->groupTypeId,
              'id' => $this->group->uuid(),
            ],
          ],
          'entity_id' => [
            'data' => [
              'type' => 'user--user',
              'id' => $user_to_add->uuid(),
            ],
          ],
        ],
      ],
    ];

    // Dispatch the request.
    $url = Url::fromRoute('jsonapi.group_relationship--' . $relationship_type->id() . '.collection.post');
    $response = $this->request('POST', $url, [
      'body' => Json::encode($doc),
      'auth' => [$allowed_user->getAccountName(), $allowed_user->pass_raw],
      'headers' => ['Content-Type' => 'application/vnd.api+json'],
    ]);

    // Assert that the request was successful.
    if ($response->getStatusCode() !== 201) {
      $this->fail('Expected 201, got ' . $response->getStatusCode() . '. Response: ' . $response->getBody()->getContents());
    }
    $this->assertEquals(201, $response->getStatusCode(), 'POST request should succeed for user with permission.');
  }

  /**
   * Tests user without permission can't create relationship via JSON:API.
   */
  public function testForbiddenCreateAccess(): void {
    // Create user & verify access.
    $forbidden_user = $this->drupalCreateUser();
    $forbidden_user->save();
    $handler = \Drupal::entityTypeManager()->getAccessControlHandler('group_relationship');
    $this->drupalLogin($forbidden_user);
    $this->assertFalse($handler->createAccess($this->groupTypeId . "-group_membership", $forbidden_user, ['group' => $this->group]));
    $this->drupalLogout();

    // Build the JSON:API request body.
    $relationship_type = GroupRelationshipType::load($this->groupTypeId . '-group_membership');
    $resource_type_name = 'group_relationship--' . $relationship_type->id();
    $user_to_add = $this->drupalCreateUser();
    $doc = [
      'data' => [
        'type' => $resource_type_name,
        'relationships' => [
          'gid' => [
            'data' => [
              'type' => 'group--' . $this->groupTypeId,
              'id' => $this->group->uuid(),
            ],
          ],
          'entity_id' => [
            'data' => [
              'type' => 'user--user',
              'id' => $user_to_add->uuid(),
            ],
          ],
        ],
      ],
    ];

    // Dispatch the request.
    $url = Url::fromRoute('jsonapi.group_relationship--' . $relationship_type->id() . '.collection.post');
    $response = $this->request('POST', $url, [
      'body' => Json::encode($doc),
      'auth' => [$forbidden_user->getAccountName(), $forbidden_user->pass_raw],
      'headers' => ['Content-Type' => 'application/vnd.api+json'],
    ]);

    // Assert that the request was forbidden.
    $this->assertEquals(403, $response->getStatusCode(), 'POST request should be forbidden for user without permission.');
  }

}
