<?php

namespace Drupal\Tests\group_jsonapi_create_access\Unit;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Cache\Context\CacheContextsManager;
use Drupal\Core\DependencyInjection\Container;
use Drupal\group_jsonapi_create_access\Access\GroupJsonapiCreateAccessCheck;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\InputBag;
use Symfony\Component\HttpFoundation\Request;

/**
 * @coversDefaultClass \Drupal\group_jsonapi_create_access\Access\GroupJsonapiCreateAccessCheck
 *
 * @group group_jsonapi_create_access
 */
class GroupJsonapiCreateAccessCheckTest extends UnitTestCase {

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $requestStack;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $cache_contexts_manager = $this->prophesize(CacheContextsManager::class);
    $cache_contexts_manager->assertValidTokens()->willReturn(TRUE);
    $cache_contexts_manager->reveal();
    $container = new Container();
    $container->set('cache_contexts_manager', $cache_contexts_manager);
    \Drupal::setContainer($container);

    $this->entityTypeManager = $this->createMock('Drupal\Core\Entity\EntityTypeManagerInterface');
    $this->requestStack = $this->createMock('Symfony\Component\HttpFoundation\RequestStack');
  }

  /**
   * Tests access when group is in JSON:API request body.
   */
  public function testAccessWithGroupInRequestBody(): void {
    // Create a mock group.
    $group = $this->createMock('Drupal\group\Entity\GroupInterface');
    $group->expects($this->any())
      ->method('id')
      ->willReturn(1);

    // Mock the group storage to return our group.
    $group_storage = $this->createMock('Drupal\Core\Entity\EntityStorageInterface');
    $group_storage->expects($this->once())
      ->method('loadByProperties')
      ->with(['uuid' => 'test-group-uuid'])
      ->willReturn([1 => $group]);

    $this->entityTypeManager->expects($this->once())
      ->method('getStorage')
      ->with('group')
      ->willReturn($group_storage);

    // Mock the access control handler to return allowed.
    $access_control_handler = $this->createMock('Drupal\Core\Entity\EntityAccessControlHandlerInterface');
    $access_control_handler->expects($this->once())
      ->method('createAccess')
      ->with('test-bundle', $this->anything(), ['group' => $group], TRUE)
      ->willReturn(AccessResult::allowed());

    $this->entityTypeManager->expects($this->once())
      ->method('getAccessControlHandler')
      ->with('group_relationship')
      ->willReturn($access_control_handler);

    // Create a JSON:API request with group UUID in body.
    $request = new Request(
      [],
      [],
      [],
      [],
      [],
      [],
      json_encode([
        'data' => [
          'type' => 'group_relationship--test-bundle',
          'relationships' => [
            'gid' => [
              'data' => [
                'type' => 'group--test',
                'id' => 'test-group-uuid',
              ],
            ],
          ],
        ],
      ])
    );
    $request->setRequestFormat('api_json');

    $this->requestStack->expects($this->once())
      ->method('getCurrentRequest')
      ->willReturn($request);

    $access_check = new GroupJsonapiCreateAccessCheck($this->entityTypeManager, $this->requestStack);

    $route = $this->getMockBuilder('Symfony\Component\Routing\Route')
      ->disableOriginalConstructor()
      ->getMock();
    $route->expects($this->any())
      ->method('getRequirement')
      ->with('_group_jsonapi_create_access')
      ->willReturn('group_relationship:test-bundle');

    $raw_parameters = new InputBag();
    $route_match = $this->createMock('Drupal\Core\Routing\RouteMatchInterface');
    $route_match->expects($this->any())
      ->method('getRawParameters')
      ->willReturn($raw_parameters);

    $account = $this->createMock('Drupal\Core\Session\AccountInterface');

    $result = $access_check->access($route, $route_match, $account);
    $this->assertTrue($result->isAllowed());
  }

  /**
   * Tests access when group is not in JSON:API request body.
   */
  public function testAccessWithoutGroupInRequestBody(): void {
    // Create a request without a group UUID in the body.
    $request = new Request(
      [],
      [],
      [],
      [],
      [],
      [],
      json_encode([
        'data' => [
          'type' => 'group_relationship--test-bundle',
        ],
      ])
    );
    $request->setRequestFormat('api_json');

    $this->requestStack->expects($this->once())
      ->method('getCurrentRequest')
      ->willReturn($request);

    $access_check = new GroupJsonapiCreateAccessCheck($this->entityTypeManager, $this->requestStack);

    $route = $this->getMockBuilder('Symfony\Component\Routing\Route')
      ->disableOriginalConstructor()
      ->getMock();
    $route->expects($this->any())
      ->method('getRequirement')
      ->with('_group_jsonapi_create_access')
      ->willReturn('group_relationship:test-bundle');

    $raw_parameters = new InputBag();
    $route_match = $this->createMock('Drupal\Core\Routing\RouteMatchInterface');
    $route_match->expects($this->any())
      ->method('getRawParameters')
      ->willReturn($raw_parameters);

    $account = $this->createMock('Drupal\Core\Session\AccountInterface');

    $result = $access_check->access($route, $route_match, $account);
    $this->assertTrue($result->isForbidden());
    $this->assertStringContainsString('Group not available from request', $result->getReason());
  }

}
