<?php

declare(strict_types=1);

namespace Drupal\group_linked_entity;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Entity\GroupRelationshipInterface;
use Drupal\group\Entity\Storage\GroupRelationshipTypeStorageInterface;
use Drupal\group\Entity\GroupRelationshipTypeInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeManager;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\Component\Utility\NestedArray;

/**
 * Helper service for various group_linked_entity features.
 */
final class Helper {

  /**
   * Object constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\group\Plugin\Group\Relation\GroupRelationTypeManager $groupRelationTypeManager
   *   The group relationship type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly GroupRelationTypeManager $groupRelationTypeManager,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly CacheBackendInterface $cache,
  ) {}

  /**
   * Returns an index of defined group/entity links.
   *
   * @code
   * $index = [
   *   'groups' => [
   *     'my_group_bundle' => [
   *       'entity_type_id' => 'node',
   *       'entity_bundle' => 'my_node_bundle',
   *       'group_relationship_type' => 'my_group_bundle-group_node-my_node_bundle',
   *       'content_plugin' => 'group_node:my_node_bundle',
   *     ]
   *   ],
   *   'entities' => [
   *     'node' => [
   *       'my_node_bundle' => [
   *         'group_bundle' => 'my_group_bundle',
   *         'group_relationship_type' => 'my_group_bundle-group_node-my_node_bundle',
   *         'content_plugin' => 'group_node:my_node_bundle',
   *       ]
   *     ],
   *   ]
   * ];
   * @endcode
   *
   * @param string|array $key
   *   The key to fetch values for. Can be an array.
   *
   * @return array|string
   *   The hole definition index or a particular definition.
   */
  public function getLinkingDefinitions($key = NULL) {
    $cid = 'group_linked_entity:linking_definitions';
    $registry = &drupal_static($cid);
    if (isset($registry)) {
      return ($key) ? NestedArray::getValue($registry, (array) $key) : $registry;
    }

    if ($cache = $this->cache->get($cid)) {
      $registry = $cache->data;
    }
    else {
      $registry = [
        'entities' => [],
        'groups' => [],
      ];
      $grt_storage = $this->entityTypeManager->getStorage('group_relationship_type');
      assert($grt_storage instanceof GroupRelationshipTypeStorageInterface);
      $group_relationship_types = $grt_storage->loadMultiple();

      $group_relationship_types = array_filter($group_relationship_types, function ($group_relationship_type) {
        return $this->getGroupEntityLinkField($group_relationship_type);
      });

      foreach ($group_relationship_types as $group_relationship_type) {
        assert($group_relationship_type instanceof GroupRelationshipTypeInterface);
        $plugin_definition = $this->groupRelationTypeManager->getDefinition($group_relationship_type->get('content_plugin'));
        $entity_type_id = $plugin_definition->getEntityTypeId();
        $entity_bundle = $plugin_definition->getEntityBundle();
        $registry['entities'][$entity_type_id][$entity_bundle] = [
          'group_bundle' => $group_relationship_type->getGroupTypeId(),
          'group_relationship_type' => $group_relationship_type->id(),
          'content_plugin' => $group_relationship_type->get('content_plugin'),
        ];
        $registry['groups'][$group_relationship_type->getGroupTypeId()] = [
          'entity_type_id' => $entity_type_id,
          'entity_bundle' => $entity_bundle,
          'group_relationship_type' => $group_relationship_type->id(),
          'content_plugin' => $group_relationship_type->get('content_plugin'),
        ];
      }
      $this->cache->set($cid, $registry, -1, ['group_type_list', 'group_relationship_type_list']);
    }

    return ($key) ? NestedArray::getValue($registry, (array) $key) : $registry;

  }

  /**
   * Checks if a group have to link with an entity.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   *
   * @return bool
   *   True if this group can be linked to an entity.
   */
  public function isLinkingEnabled(GroupInterface $group) {
    $index = $this->getLinkingDefinitions();
    return array_key_exists($group->bundle(), $index['groups']);
  }

  /**
   * Returns the linked entity for this group.
   *
   * Or false if not found.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   *
   * @return bool|ContentEntityInterface
   *   The entity linked to the given group.
   */
  public function getLinkedEntity(GroupInterface $group) {
    if ($group->isNew()) {
      return FALSE;
    }
    $index = $this->getLinkingDefinitions('groups');
    if (array_key_exists($group->bundle(), $index)) {
      if ($relations = $this->entityTypeManager->getStorage('group_relationship')->loadByProperties([
        'gid' => $group->id(),
        'type' => $index[$group->bundle()]['group_relationship_type'],
        'group_entity_link' => TRUE,
      ])) {
        $relation = array_shift($relations);
        assert($relation instanceof GroupRelationshipInterface);
        return $relation->getEntity();
      }
    }

    return FALSE;
  }

  /**
   * Returns the linked entity for this group ID.
   *
   * Or false if not found.
   *
   * @param int $gid
   *   The group entity id.
   *
   * @return bool|ContentEntityInterface
   *   The entity linked to the given group.
   */
  public function getLinkedEntityById(int $gid) {
    if($group = $this->entityTypeManager->getStorage('group')->load($gid)) {
      return $this->getLinkedEntity($group);
    }
    return FALSE;
  }

  /**
   * Create a linked entity for the given group.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   */
  public function createLinkedEntity(GroupInterface $group) {
    $index = $this->getLinkingDefinitions('groups');
    if (array_key_exists($group->bundle(), $index)) {
      $entity_type_id = $index[$group->bundle()]['entity_type_id'];
      $entity_bundle = $index[$group->bundle()]['entity_bundle'];
      $content_plugin_id = $index[$group->bundle()]['content_plugin'];

      $bundle_key = $this->entityTypeManager->getDefinition($entity_type_id)->getKey('bundle');
      $label_key = $this->entityTypeManager->getDefinition($entity_type_id)->getKey('label');

      $entity = $this->entityTypeManager->getStorage($entity_type_id)->create([
        $bundle_key => $entity_bundle,
        $label_key => $group->label(),
      ]);
      $entity->save();

      // Attach entity.
      $group->addRelationship($entity, $content_plugin_id, ['group_entity_link' => TRUE]);

    }
  }

  /**
   * Returns the linking group for this entity.
   *
   * Or false if none found.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity.
   *
   * @return bool|GroupInterface
   *   The group the entity is linked to or false.
   */
  public function getLinkingGroup(ContentEntityInterface $entity) {
    $index = $this->getLinkingDefinitions('entities');
    if (
      !$entity->isNew() &&
      array_key_exists($entity->getEntityTypeId(), $index) &&
      array_key_exists($entity->bundle(), $index[$entity->getEntityTypeId()])
    ) {
      if ($relationships = $this->entityTypeManager->getStorage('group_relationship')->loadByProperties([
        'entity_id' => $entity->id(),
        'group_entity_link' => 1,
      ])) {

        $group_relationship_types = [];
        foreach ($index[$entity->getEntityTypeId()] as $data) {
          $group_relationship_types[] = $data['group_relationship_type'];
        }

        $relationships = array_filter($relationships, function (ContentEntityInterface $relationship) use ($group_relationship_types) {
          if (!in_array($relationship->bundle(), $group_relationship_types)) {
            return FALSE;
          }
          return TRUE;
        });

        if (empty($relationships)) {
          return FALSE;
        }

        $relationship = reset($relationships);
        assert($relationship instanceof GroupRelationshipInterface);
        return $relationship->getGroup();
      }
    }
    return FALSE;
  }

  /**
   * Get available content entity types.
   *
   * @return array
   *   An array of available content entity types that
   *   could be used as links to groups.
   */
  public function getAvailableContentEntityTypes() {
    $definitions = [];
    if ($definitions = $this->groupRelationTypeManager->getDefinitions()) {
      $definitions = array_filter($definitions, function ($definition) {
        return ('node' == $definition->getEntityTypeId() && $this->moduleHandler->moduleExists('gnode')) ||
        ('taxonomy_term' == $definition->getEntityTypeId() && $this->moduleHandler->moduleExists('gterm'));
      });
    }
    return $definitions;
  }

  /**
   * Get the linking field if it exists.
   *
   * This field is used to identify the link relationship.
   *
   * @param \Drupal\group\Entity\GroupRelationshipTypeInterface $group_relationship_type
   *   The group relationship type object.
   *
   * @return mixed
   *   Returns the linking field if it exists.
   */
  private function getGroupEntityLinkField(GroupRelationshipTypeInterface $group_relationship_type) {
    $field = FieldConfig::loadByName('group_relationship', $group_relationship_type->id(), 'group_entity_link');
    return $field;
  }

}
