<?php

namespace Drupal\group_linked_entity\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\group\Entity\GroupTypeInterface;
use Drupal\group\Entity\GroupRelationshipTypeInterface;
use Drupal\group\Entity\Storage\GroupRelationshipTypeStorageInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeManager;
use Drupal\group_linked_entity\Helper;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Hook implementations.
 */
class DefinitionsHooks {

  use StringTranslationTrait;

  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly ModuleHandlerInterface $moduleHandler,
    protected readonly GroupRelationTypeManager $groupRelationTypeManager,
    protected readonly Helper $helper,
    protected readonly MessengerInterface $messenger,
  ) {}

  /**
   * Implements hook_form_FORM_ID_alter().
   */
  #[Hook('form_group_type_form_alter')]
  public function groupTypeFormSettings(array &$form, FormStateInterface $form_state) {
    if ($this->moduleHandler->moduleExists('gnode') || $this->moduleHandler->moduleExists('gterm')) {
      $form['group_linked_entity'] = [
        '#type' => 'details',
        '#title' => 'Group linked entity',
        '#group' => 'additional_settings',
      ];

      $type = $form_state->getFormObject()->getEntity();

      $form['group_linked_entity']['entity_type'] = [
        '#type' => 'select',
        '#title' => $this->t('Content type'),
        '#options' => $this->getAvailableEntityTypes(),
        '#empty_option' => t('None'),
        '#empty_value' => '',
        '#default_value' => $type->getThirdPartySetting('group_linked_entity', 'entity_type', FALSE),
        '#description' => t('Link groups of this kind with entities of the selected entity type. <br>
        For each group created, an entity with the same label will be created and attached to the group.'),
        '#multiple' => FALSE,
      ];

      $form['#entity_builders'][] = [get_called_class(), 'submitForm'];
    }

  }

  /**
   * Implements hook_entity_update().
   *
   * Automatically create the group relationship type.
   */
  #[Hook('entity_insert')]
  #[Hook('entity_update')]
  public function ensureGroupRelationshipType(EntityInterface $group_type) {
    // Only act on GroupType entities.
    if (!$group_type instanceof GroupTypeInterface) {
      return;
    }
    if ($plugin_id = $group_type->getThirdPartySetting('group_linked_entity', 'entity_type', FALSE)) {

      if (!in_array($plugin_id, $this->groupRelationTypeManager->getInstalledIds($group_type))) {
        // Create Group relationship type if it does not exist yet.
        $this->createGroupRelationType($group_type, $plugin_id);
      }
      else {
        $grt_storage = $this->entityTypeManager->getStorage('group_relationship_type');
        assert($grt_storage instanceof GroupRelationshipTypeStorageInterface);
        $relationship_bundle = $grt_storage->getRelationshipTypeId($group_type->id(), $plugin_id);
        $group_relationship_type = $grt_storage->load($relationship_bundle);

        // 1. Ensure field "group_entity_link" is added to relation type.
        $this->attachGroupEntityLinkField($group_relationship_type);

        // 2. Warn the user if plugin already installed.
        $this->messenger->addWarning($this->t('Warning : This content type is already installed on this group type.
        There may be already contents that are attached to groups of this type.'));

      }
    }
  }

  /**
   * Form submit for form_group_type_form_alter.
   */
  public static function submitForm($entity_type, GroupTypeInterface $group_type, &$form, FormStateInterface $form_state) {
    $group_type->setThirdPartySetting('group_linked_entity', 'entity_type', $form_state->getValue('entity_type'));
  }

  /**
   * Get available entity types.
   *
   * @return array
   *   An array of available content entity types.
   */
  protected function getAvailableEntityTypes() {
    $options = [];
    if ($definitions = $this->helper->getAvailableContentEntityTypes()) {
      foreach ($definitions as $plugin_id => $definition) {
        $label = $definition->getLabel()->getArguments();
        $value = reset($label) . " ({$definition->getEntityTypeId()})";
        $options[$plugin_id] = $value;
      }
    }
    return $options;
  }

  /**
   * Create group relationship for the given group type.
   *
   * Create group relationship for the given group type and attaches
   * a group_entity_field field to it.
   *
   * @param \Drupal\group\Entity\GroupTypeInterface $group_type
   *   Group type object.
   * @param string $plugin_id
   *   A plugin id string.
   */
  protected function createGroupRelationType(GroupTypeInterface $group_type, $plugin_id) {
    $group_relationship_type = $this->doCreateGroupRelationType($group_type, $plugin_id);
    $this->attachGroupEntityLinkField($group_relationship_type);
  }

  /**
   * Create group relationship for the given group type.
   *
   * @param \Drupal\group\Entity\GroupTypeInterface $group_type
   *   Group type object.
   * @param string $plugin_id
   *   A plugin id string.
   *
   * @return \Drupal\group\Entity\GroupRelationshipTypeInterface
   *   The created group relationship type.
   */
  protected function doCreateGroupRelationType(GroupTypeInterface $group_type, $plugin_id) {
    $storage = $this->entityTypeManager->getStorage('group_relationship_type');
    assert($storage instanceof GroupRelationshipTypeStorageInterface);
    $group_relationship_type = $storage->createFromPlugin($group_type, $plugin_id, [
      'group_cardinality' => 1,
    ]);
    $group_relationship_type->save();
    return $group_relationship_type;
  }

  /**
   * Attach the linking field to the given group_relationship_type.
   *
   * This field is used to identify the link relationship.
   *
   * @param \Drupal\group\Entity\GroupRelationshipTypeInterface $group_relationship_type
   *   The group relationship type object.
   */
  protected function attachGroupEntityLinkField(GroupRelationshipTypeInterface $group_relationship_type) {
    $field_storage = FieldStorageConfig::loadByName('group_relationship', 'group_entity_link');
    $field = FieldConfig::loadByName('group_relationship', $group_relationship_type->id(), 'group_entity_link');

    if (empty($field)) {
      $field = FieldConfig::create([
        'field_storage' => $field_storage,
        'bundle' => $group_relationship_type->id(),
        'label' => 'Group entity link',
      ]);
      $field->save();
    }

  }

}
