<?php

namespace Drupal\group_linked_entity\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Entity\GroupRelationshipInterface;
use Drupal\group_linked_entity\Helper;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\ContentEntityType;

/**
 * Hook implementations.
 */
class EntityHooks {

  use StringTranslationTrait;

  public function __construct(
    protected readonly Helper $helper,
  ) {}

  /**
   * Implements hook_entity_update().
   *
   * Ensure a linked entity is created for each group.
   */
  #[Hook('entity_insert')]
  #[Hook('entity_update')]
  public function ensureLinkedEntityExists(EntityInterface $group) {
    // Only act on Group entities.
    if (!$group instanceof GroupInterface) {
      return;
    }

    // Only act on Group entities that have bounding.
    if (!$this->helper->isLinkingEnabled($group)) {
      return;
    }

    if (!$this->helper->getLinkedEntity($group)) {
      $this->helper->createLinkedEntity($group);
    }

  }

  /**
   * Implements hook_entity_delete().
   *
   * Propagate delete linking entity when a link is deleted.
   */
  #[Hook('entity_delete')]
  public function deleteLinkedEntity(EntityInterface $group_relationship) {
    // Only act on group relationships entities.
    if (!$group_relationship instanceof GroupRelationshipInterface) {
      return;
    }

    // Only act on group relationships that are links.
    if (!$group_relationship->hasField('group_entity_link') || $group_relationship->group_entity_link->isEmpty() || !(bool) $group_relationship->group_entity_link->getString()) {
      return;
    }

    if ($entity = $group_relationship->getEntity()) {
      $entity->delete();
    }

  }

  /**
   * Implements hook_entity_predelete().
   */
  #[Hook('entity_predelete')]
  public function preDeleteLinkedEntity(EntityInterface $entity) {
    // Only act on group ContentEntityInterface entities.
    if (!$entity instanceof ContentEntityInterface) {
      return;
    }

    if (!$entity->hasField('linked_group') || $entity->get('linked_group')->isEmpty()) {
      return;
    }

    throw new \Exception('This entity linked to a group and cannot be deleted.');

  }

  /**
   * Implements hook_entity_access().
   *
   * Should be in some validation layer (constraints for instance),
   * but not possible right now with drupal.
   */
  #[Hook('entity_access')]
  public function entityAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    if ($operation === 'delete' && $entity instanceof ContentEntityInterface) {
      return AccessResult::forbiddenIf(
        ($entity->hasField('linked_group') && !$entity->get('linked_group')->isEmpty()),
      'This entity is linked to a group and cannot be deleted.');
    }
    return AccessResult::neutral();
  }

  /**
   * Implements hook_entity_type_alter().
   *
   * Declare entity constraints.
   */
  #[Hook('entity_type_alter')]
  public function addConstraints(array &$entity_types) {
    // Constraints for group_relationships.
    if (!empty($entity_types['group_relationship'])) {
      // Prevent linking with multiple entities.
      $entity_types['group_relationship']->addConstraint('UniqueLinkingWithEntity');
      // Prevents linking with multiple groups.
      $entity_types['group_relationship']->addConstraint('UniqueLinkingWithGroup');
    }
  }

  /**
   * Implements hook_entity_bundle_field_info().
   *
   * Attach computed fields to groups.
   */
  #[Hook('entity_bundle_field_info')]
  public function attachGroupFields(EntityTypeInterface $group_type, $bundle, array $base_field_definitions) {

    if (!$group_type instanceof ContentEntityType) {
      return [];
    }

    if ($group_type->id() != 'group') {
      return [];
    }

    $index = $this->helper->getLinkingDefinitions();
    if (!array_key_exists($bundle, $index['groups'])) {
      return [];
    }

    $entity_data = $index['groups'][$bundle];

    $fields = [];

    // Related entity.
    $fields['linked_entity'] = BaseFieldDefinition::create('entity_reference')
      ->setTargetEntityTypeId($entity_data['entity_type_id'])
      ->setTargetBundle(NULL)
      ->setName($entity_data['entity_type_id'])
      ->setLabel('Linked content')
      ->setComputed(TRUE)
      ->setCustomStorage(TRUE)
      ->setReadOnly(TRUE)
      ->setSettings([
        'target_type' => $entity_data['entity_type_id'],
        'handler' => 'default:' . $entity_data['entity_type_id'],
        'handler_settings' => [
          'target_bundles' => [$entity_data['entity_bundle'] => $entity_data['entity_bundle']],
          'auto_create' => FALSE,
          'auto_create_bundle' => '',
        ],
      ])
      ->setClass('\Drupal\group_linked_entity\Field\LinkedEntity')
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', ['region' => 'hidden']);

    return $fields;
  }

  /**
   * Implements hook_entity_bundle_field_info().
   *
   * Attach computed fields to entities.
   */
  #[Hook('entity_bundle_field_info')]
  public function attachLinkedEntityFields(EntityTypeInterface $entity_type, $bundle, array $base_field_definitions) {

    $index = $this->helper->getLinkingDefinitions();

    if (!array_key_exists($entity_type->id(), $index['entities'])) {
      return [];
    }

    if (!array_key_exists($bundle, $index['entities'][$entity_type->id()])) {
      return [];
    }

    $group_bundle = $index['entities'][$entity_type->id()][$bundle]['group_bundle'];

    // Related group.
    $fields['linked_group'] = BaseFieldDefinition::create('entity_reference')
      ->setTargetEntityTypeId('group')
      ->setTargetBundle(NULL)
      ->setName('group')
      ->setLabel('Linked group')
      ->setComputed(TRUE)
      ->setCustomStorage(TRUE)
      ->setReadOnly(TRUE)
      ->setSettings([
        'target_type' => 'group',
        'handler' => 'default:group',
        'handler_settings' => [
          'target_bundles' => [$group_bundle => $group_bundle],
          'auto_create' => FALSE,
          'auto_create_bundle' => '',
        ],
      ])
      ->setClass('\Drupal\group_linked_entity\Field\LinkedGroup')
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', ['region' => 'hidden']);

    return $fields;
  }

}
