<?php

declare(strict_types=1);

namespace Drupal\group_media_bulk_upload;

use Drupal\Core\Utility\Token;
use Drupal\Component\Render\PlainTextOutput;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Mime\MimeTypeGuesserInterface;
use Symfony\Component\Validator\ConstraintViolationListInterface;
use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Environment;
use Drupal\file\Validation\FileValidatorInterface;
use Drupal\file\FileInterface;
use Drupal\group\Entity\GroupRelationshipTypeInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Entity\Storage\GroupRelationshipTypeStorage;
use Drupal\media\MediaTypeInterface;

/**
 * Media manager.
 */
class BulkMediaManager {

  /**
   * Constructs a BulkMediaManager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\file\Validation\FileValidatorInterface $fileValidator
   *   The file validator.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $privateTempStoreFactory
   *   The private temp store factory.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system.
   * @param \Symfony\Component\Mime\MimeTypeGuesserInterface $mimeTypeGuesser
   *   The mime type guesser.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly FileValidatorInterface $fileValidator,
    protected readonly AccountInterface $currentUser,
    protected readonly PrivateTempStoreFactory $privateTempStoreFactory,
    protected readonly FileSystemInterface $fileSystem,
    #[Autowire(service: 'file.mime_type.guesser')]
    protected readonly MimeTypeGuesserInterface $mimeTypeGuesser,
    protected readonly Token $token,
    protected readonly RequestStack $requestStack,
  ) {
    // No op.
  }

  /**
   * Get media types available for upload to a given group.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   *
   * @return \Drupal\group\Entity\GroupRelationshipTypeInterface[]
   *   Media group relation types.
   */
  public function getMediaTypes(GroupInterface $group): array {
    // Load media plugins.
    $relationship_type_storage = $this->entityTypeManager->getStorage('group_relationship_type');
    assert($relationship_type_storage instanceof GroupRelationshipTypeStorage);

    $relationship_types = $relationship_type_storage->loadByGroupType($group->getGroupType());
    return array_filter($relationship_types, function ($type) {
      if (!str_starts_with($type->getPluginId(), 'group_media:')) {
        return FALSE;
      }

      // Load the media type and only include if the source field is a file or
      // image type.
      if (!$media_type = $this->getMediaTypeForGroupRelationshipType($type)) {
        return FALSE;
      }

      assert($media_type instanceof MediaTypeInterface);
      return in_array($media_type->getSource()->getSourceFieldDefinition($media_type)->getType(), ['image', 'file']);
    });
  }

  /**
   * Gets the group media type for a given file URI.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group where the file is being uploaded.
   * @param string $uri
   *   The file URI.
   *
   * @return string|null
   *   The group content type ID, or NULL if none found.
   */
  public function getGroupMediaTypeForFile(GroupInterface $group, string $uri): ?string {
    foreach ($this->getMediaTypes($group) as $type) {
      if (!$media_type = $this->getMediaTypeForGroupRelationshipType($type)) {
        continue;
      }

      // Permission to create the media.
      if (!$type->access('create', $this->currentUser, FALSE)) {
        continue;
      }

      assert($media_type instanceof MediaTypeInterface);
      $extensions = $media_type->getSource()->getSourceFieldDefinition($media_type)->getSetting('file_extensions');
      if (!$extensions) {
        continue;
      }

      $regex = '/\.(' . preg_replace('/ +/', '|', preg_quote($extensions)) . ')$/i';
      if (preg_match($regex, $uri)) {
        return $type->id();
      }
    }

    return NULL;
  }

  /**
   * Validate a media upload.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file to validate.
   * @param string $type
   *   The group media type ID.
   *
   * @return \Symfony\Component\Validator\ConstraintViolationListInterface
   *   An array of error messages, or an empty array if valid.
   */
  public function validateMediaUpload(FileInterface $file, string $type): ConstraintViolationListInterface {
    $group_relationship_type = $this->entityTypeManager
      ->getStorage('group_relationship_type')
      ->load($type);
    assert($group_relationship_type instanceof GroupRelationTypeInterface);

    $media_type = $this->getMediaTypeForGroupRelationshipType($group_relationship_type);
    $source = $media_type->getSource()->getSourceFieldDefinition($media_type);
    $field_settings = $source->getSettings();

    $upload_validators = [];

    $max_filesize = Bytes::toNumber(Environment::getUploadMaxSize());
    if (!empty($field_settings['max_filesize'])) {
      $max_filesize = min($max_filesize, Bytes::toNumber($field_settings['max_filesize']));
    }

    // There is always a file size limit due to the PHP server limit.
    $upload_validators['FileSizeLimit'] = ['fileLimit' => $max_filesize];

    if ($source->getType() == 'image') {
      $upload_validators['FileIsImage'] = [];
      if ($field_settings['max_resolution'] || $field_settings['min_resolution']) {
        $upload_validators['GroupMediaBulkUploadFileImageDimensions'] = [
          'maxDimensions' => $field_settings['max_resolution'],
          'minDimensions' => $field_settings['min_resolution'],
        ];
      }
    }

    return $this->fileValidator->validate($file, $upload_validators);
  }

  /**
   * Process an uploaded file.
   *
   * @param string $type
   *   The group media content type.
   * @param string $uri
   *   The URI to the uploaded file.
   */
  public function processUpload(string $type, string $uri): FileInterface {
    $group_relationship_type = $this->entityTypeManager
      ->getStorage('group_relationship_type')
      ->load($type);
    assert($group_relationship_type instanceof GroupRelationTypeInterface);

    // Copy the file to the destination directory per field settings.
    $media_type = $this->getMediaTypeForGroupRelationshipType($group_relationship_type);
    $field = $media_type->getSource()->getSourceFieldDefinition($media_type);
    $settings = $field->getSettings();
    $destination = trim($settings['file_directory'], '/');
    $destination = PlainTextOutput::renderFromHtml($this->token->replace($destination));
    $destination_dir = $settings['uri_scheme'] . '://' . $destination;
    $this->fileSystem->prepareDirectory($destination_dir, FileSystemInterface::CREATE_DIRECTORY);
    $destination = $destination_dir . basename($uri);
    $uri = $this->fileSystem->move($uri, $destination);

    /** @var \Drupal\file\FileInterface $file */
    $file = $this->entityTypeManager->getStorage('file')->create([
      'uri' => $uri,
      'uid' => $this->currentUser->id(),
    ]);
    $file->setMimeType($this->mimeTypeGuesser->guessMimeType($uri));
    $file->setSize(@filesize($this->fileSystem->realPath($uri)));
    $file->setTemporary();
    return $file;
  }

  /**
   * Get the media type for a given group relationship type.
   *
   * @param \Drupal\group\Entity\GroupRelationshipTypeInterface $group_relationship_type
   *   The group relationship type.
   *
   * @return \Drupal\media\MediaTypeInterface|null
   *   The media type, or NULL if not found.
   */
  protected function getMediaTypeForGroupRelationshipType(GroupRelationshipTypeInterface $group_relationship_type): ?MediaTypeInterface {
    $media_type = $this->entityTypeManager->getStorage('media_type')
      ->load(substr($group_relationship_type->getPluginId(), strlen('group_media:')));
    assert($media_type instanceof MediaTypeInterface);
    return $media_type;
  }

  /**
   * Set uploaded files in the temp store.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   * @param array $file_ids
   *   The file IDs.
   * @param string|null $redirect
   *   (optional) A redirect URL to return to when done.
   *
   * @return $this
   */
  public function setUploadedFiles(GroupInterface $group, array $file_ids, ?string $redirect = NULL) : self {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    $temp_store->set('group', $group->id());
    $temp_store->set('file_ids', $file_ids);
    if ($redirect) {
      $temp_store->set('redirect', $redirect);
    }
    else {
      $temp_store->delete('redirect');
    }
    return $this;
  }

  /**
   * Clear the uploaded files from the temp store and delete them.
   *
   * @return $this
   */
  public function clearUploadedFiles() : self {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    $fids = $temp_store->get('file_ids');
    $files = $this->entityTypeManager->getStorage('file')->loadMultiple($fids);
    $this->entityTypeManager->getStorage('file')->delete($files);
    $temp_store->set('file_ids', []);
    return $this;
  }

  /**
   * Move to the next file in the bulk upload process.
   *
   * @return $this
   */
  public function nextUploadedFile() :self {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    $fids = $temp_store->get('file_ids');
    if ($fid = array_key_first($fids)) {
      unset($fids[$fid]);
      $temp_store->set('file_ids', $fids);
    }
    return $this;
  }

  /**
   * Get the next redirect URL for the bulk upload process.
   *
   * @return \Drupal\Core\Url|null
   *   The redirect URL, or NULL if none found.
   */
  public function getNextBulkUploadRedirect(): ?Url {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');

    $gid = $temp_store->get('group');
    $fids = $temp_store->get('file_ids');
    $fid = array_key_first($fids);

    if ($gid && $fid) {
      $relationship_type_id = $fids[$fid];
      $relationship_type_storage = $this->entityTypeManager->getStorage('group_relationship_type');
      assert($relationship_type_storage instanceof GroupRelationshipTypeStorage);
      $relationship_type = $relationship_type_storage->load($relationship_type_id);
      assert($relationship_type instanceof GroupRelationshipTypeInterface);
      $plugin = $relationship_type->getPluginId();

      return Url::fromRoute('entity.group_relationship.create_form', [
        'group' => $gid,
        'plugin_id' => $plugin,
        'bulk_fid' => $fid,
      ]);
    }

    if ($redirect = $temp_store->get('redirect')) {
      return Url::fromUri('internal:' . $redirect);
    }

    return NULL;
  }

  /**
   * Obtain the bulk upload file from the 'bulk_fid' query parameter.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity or NULL available.
   */
  public function getBulkUploadFile(): ?FileInterface {
    if (!$fid = $this->requestStack->getCurrentRequest()->query->get('bulk_fid')) {
      return NULL;
    }

    if (!$file = $this->entityTypeManager->getStorage('file')->load($fid)) {
      return NULL;
    }

    assert($file instanceof FileInterface);
    return $file;
  }

}
