<?php

namespace Drupal\group_media_bulk_upload;

use Drupal\group\Entity\GroupRelationshipTypeInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeManagerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Handles access control for group media bulk uploads.
 */
class GroupMediaUploadAccess {

  /**
   * Constructs a GroupMediaUploadAccess object.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\group_media_bulk_upload\MediaTypeManager $mediaTypeManager
   *   The media type manager service.
   * @param \Drupal\group\Plugin\Group\Relation\GroupRelationTypeManagerInterface $groupRelationTypeManager
   *   The group relation type manager service.
   */
  public function __construct(
    protected readonly AccountInterface $currentUser,
    protected readonly MediaTypeManager $mediaTypeManager,
    #[Autowire(service: 'group_relation_type.manager')]
    protected readonly GroupRelationTypeManagerInterface $groupRelationTypeManager,
  ) {
    // No op.
  }

  /**
   * Determines if a user has access to upload media to a group.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   (optional) The user account to check access for. If NULL, the current
   *   user is used.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function groupUploadAccess(GroupInterface $group, ?AccountInterface $account = NULL): AccessResultInterface {
    if (!$account) {
      $account = $this->currentUser;
    }

    $access = AccessResult::neutral();
    foreach ($this->mediaTypeManager->getMediaTypes($group) as $type) {
      $access = $access->orIf($this->groupRelationshipTypeAccess($group, $type, $account));
    }

    return $access;
  }

  /**
   * Checks access for a specific group relationship type.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   * @param \Drupal\group\Entity\GroupRelationshipTypeInterface $type
   *   The relationship type.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   (optional) The user account to check access for. If NULL, the current
   *   user is used.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function groupRelationshipTypeAccess(GroupInterface $group, GroupRelationshipTypeInterface $type, ?AccountInterface $account = NULL): AccessResultInterface {
    if (!$account) {
      $account = $this->currentUser;
    }
    $access_handler = $this->groupRelationTypeManager->getAccessControlHandler($type->getPluginId());
    return $access_handler->entityCreateAccess($group, $account, TRUE);
  }

}
