<?php

namespace Drupal\group_media_bulk_upload;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Entity\GroupRelationshipTypeInterface;
use Drupal\group\Entity\Storage\GroupRelationshipTypeStorage;
use Drupal\media\MediaTypeInterface;

/**
 * Class to manage media types for group bulk upload.
 */
class MediaTypeManager {

  /**
   * Constructs a MediaTypeManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(protected readonly EntityTypeManagerInterface $entityTypeManager) {
    // No op.
  }

  /**
   * Get media types available for upload to a given group.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   *
   * @return \Drupal\group\Entity\GroupRelationshipTypeInterface[]
   *   Media group relation types.
   */
  public function getMediaTypes(GroupInterface $group): array {
    // Load media plugins.
    $relationship_type_storage = $this->entityTypeManager->getStorage('group_relationship_type');
    assert($relationship_type_storage instanceof GroupRelationshipTypeStorage);

    $relationship_types = $relationship_type_storage->loadByGroupType($group->getGroupType());
    return array_filter($relationship_types, function ($type) {
      if (!str_starts_with($type->getPluginId(), 'group_media:')) {
        return FALSE;
      }

      // Load the media type and only include if the source field is a file or
      // image type.
      if (!$media_type = $this->getMediaTypeForGroupRelationshipType($type)) {
        return FALSE;
      }

      assert($media_type instanceof MediaTypeInterface);
      return in_array($media_type->getSource()->getSourceFieldDefinition($media_type)->getType(), ['image', 'file']);
    });
  }

  /**
   * Get the media type for a given group relationship type.
   *
   * @param \Drupal\group\Entity\GroupRelationshipTypeInterface $group_relationship_type
   *   The group relationship type.
   *
   * @return \Drupal\media\MediaTypeInterface|null
   *   The media type, or NULL if not found.
   */
  public function getMediaTypeForGroupRelationshipType(GroupRelationshipTypeInterface $group_relationship_type): ?MediaTypeInterface {
    $media_type = $this->entityTypeManager->getStorage('media_type')
      ->load(substr($group_relationship_type->getPluginId(), strlen('group_media:')));
    assert($media_type instanceof MediaTypeInterface);
    return $media_type;
  }

}
