<?php

namespace Drupal\group_media_bulk_upload\Plugin\Validation\Constraint;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\file\Plugin\Validation\Constraint\BaseFileConstraintValidator;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validator for the FileImageDimensionsConstraint.
 *
 * Similar to core FileImageDimensionsValidator but removed the image resize.
 */
class FileImageDimensionsConstraintValidator extends BaseFileConstraintValidator implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * Creates a new FileImageDimensionsConstraintValidator.
   *
   * @param \Drupal\Core\Image\ImageFactory $imageFactory
   *   The image factory.
   */
  public function __construct(protected readonly ImageFactory $imageFactory) {
    // No op.
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static($container->get('image.factory'));
  }

  /**
   * {@inheritdoc}
   */
  public function validate(mixed $value, Constraint $constraint): void {
    $file = $this->assertValueIsFile($value);
    if (!$constraint instanceof FileImageDimensionsConstraint) {
      throw new UnexpectedTypeException($constraint, FileImageDimensionsConstraint::class);
    }

    $image = $this->imageFactory->get($file->getFileUri());
    if (!$image->isValid()) {
      return;
    }

    $maxDimensions = $constraint->maxDimensions;
    if ($maxDimensions) {
      // Check that it is smaller than the given dimensions.
      [$width, $height] = explode('x', $maxDimensions);
      if ($image->getWidth() > $width || $image->getHeight() > $height) {
        $this->context->addViolation($constraint->messageImageTooLarge,
          [
            '%dimensions' => $maxDimensions,
            '%width' => $image->getWidth(),
            '%height' => $image->getHeight(),
          ]);
      }
    }

    $minDimensions = $constraint->minDimensions;
    if ($minDimensions) {
      // Check that it is larger than the given dimensions.
      [$width, $height] = explode('x', $minDimensions);
      if ($image->getWidth() < $width || $image->getHeight() < $height) {
        $this->context->addViolation($constraint->messageImageTooSmall,
          [
            '%dimensions' => $minDimensions,
            '%width' => $image->getWidth(),
            '%height' => $image->getHeight(),
          ]);
      }
    }
  }

}
