<?php

namespace Drupal\group_media_bulk_upload;

use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\file\FileInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Entity\GroupRelationshipTypeInterface;
use Drupal\group\Entity\Storage\GroupRelationshipTypeStorage;

/**
 * Upload manager service.
 */
class UploadManager {

  const STATUS_CANCELLED = 'cancelled';

  const STATUS_COMPLETE = 'complete';

  const STATUS_PENDING = 'pending';

  /**
   * Constructs a UploadManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $privateTempStoreFactory
   *   The private temp store factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected PrivateTempStoreFactory $privateTempStoreFactory,
    protected RequestStack $requestStack,
  ) {
    // No op.
  }

  /**
   * Set uploaded files in the temp store.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   * @param array $file_ids
   *   The file IDs.
   * @param string|null $redirect
   *   (optional) A redirect URL to return to when done.
   *
   * @return $this
   */
  public function setUploadedFiles(GroupInterface $group, array $file_ids, ?string $redirect = NULL) : self {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    $temp_store->set('group', $group->id());

    $file_info = [];
    foreach ($file_ids as $file_id => $group_relationship_type) {
      $file_info[$file_id] = [
        'fid' => $file_id,
        'group_relationship_type' => $group_relationship_type,
        'status' => static::STATUS_PENDING,
      ];
    }
    $temp_store->set('file_info', $file_info);

    if ($redirect) {
      $temp_store->set('redirect', $redirect);
    }
    else {
      $temp_store->delete('redirect');
    }

    return $this;
  }

  /**
   * Clear the uploaded files from the temp store and delete them.
   *
   * @return $this
   */
  public function clearUploadedFiles() : self {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    $fids = $temp_store->get('file_ids');
    $files = $this->entityTypeManager->getStorage('file')->loadMultiple($fids);
    $this->entityTypeManager->getStorage('file')->delete($files);
    $temp_store->set('file_info', []);
    return $this;
  }

  /**
   * Set the status of a file in the bulk upload process.
   *
   * @param int $fid
   *   The file ID.
   * @param string $success
   *   The status to set.
   *
   * @return $this
   */
  public function setFileStatus(int $fid, string $success): self {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    $file_info = $temp_store->get('file_info');

    if (isset($file_info[$fid])) {
      $file_info[$fid]['status'] = $success;
      $temp_store->set('file_info', $file_info);
    }

    return $this;
  }

  /**
   * Get the next file ID to be uploaded.
   *
   * @return array|null
   *   The next file ID, or NULL if none found.
   */
  protected function getNextUpload(): ?array {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    $fids = $temp_store->get('file_info');
    foreach ($fids as $item) {
      if (($item['status'] ?? NULL) == static::STATUS_PENDING) {
        return $item;
      }
    }
    return NULL;
  }

  /**
   * Get the next redirect URL for the bulk upload process.
   *
   * @return \Drupal\Core\Url|null
   *   The redirect URL, or NULL if none found.
   */
  public function getNextBulkUploadRedirect(): ?Url {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');

    $gid = $temp_store->get('group');
    $next_upload = $this->getNextUpload();

    if ($gid && $next_upload) {
      $relationship_type_id = $next_upload['group_relationship_type'];

      $relationship_type_storage = $this->entityTypeManager->getStorage('group_relationship_type');
      assert($relationship_type_storage instanceof GroupRelationshipTypeStorage);
      $relationship_type = $relationship_type_storage->load($relationship_type_id);
      assert($relationship_type instanceof GroupRelationshipTypeInterface);
      $plugin = $relationship_type->getPluginId();

      return Url::fromRoute('entity.group_relationship.create_form', [
        'group' => $gid,
        'plugin_id' => $plugin,
        'bulk_fid' => $next_upload['fid'],
      ]);
    }

    if ($redirect = $temp_store->get('redirect')) {
      return Url::fromUri('internal:' . $redirect);
    }

    return NULL;
  }

  /**
   * Obtain the bulk upload file from the 'bulk_fid' query parameter.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity or NULL available.
   */
  public function getBulkUploadFile(): ?FileInterface {
    if (!$fid = $this->requestStack->getCurrentRequest()->query->get('bulk_fid')) {
      return NULL;
    }

    if (!$file = $this->entityTypeManager->getStorage('file')->load($fid)) {
      return NULL;
    }

    assert($file instanceof FileInterface);
    return $file;
  }

  /**
   * Get upload info from the temp store.
   *
   * @return array
   *   The upload info.
   */
  public function getUploadInfo(): array {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    return $temp_store->get('file_info') ?? [];
  }

}
