<?php

namespace Drupal\group_media_bulk_upload;

use Drupal\Component\Uuid\UuidInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\file\FileInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Entity\GroupRelationshipTypeInterface;
use Drupal\group\Entity\Storage\GroupRelationshipTypeStorage;

/**
 * Upload manager service.
 */
class UploadManager {

  const STATUS_CANCELLED = 'cancelled';

  const STATUS_COMPLETE = 'complete';

  const STATUS_PENDING = 'pending';

  /**
   * Constructs a UploadManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $privateTempStoreFactory
   *   The private temp store factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Component\Uuid\UuidInterface $uuidGenerator
   *   The UUID generator.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected PrivateTempStoreFactory $privateTempStoreFactory,
    protected RequestStack $requestStack,
    protected UuidInterface $uuidGenerator,
  ) {
    // No op.
  }

  /**
   * Set uploaded files in the temp store.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   * @param array $file_ids
   *   The file IDs.
   * @param string|null $redirect
   *   (optional) A redirect URL to return to when done.
   *
   * @return string
   *   The upload identifier.
   */
  public function initializeUpload(GroupInterface $group, array $file_ids, ?string $redirect = NULL): string {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');

    $info = [
      'group' => $group->id(),
      'files' => [],
      'redirect' => $redirect,
    ];

    foreach ($file_ids as $file_id => $group_relationship_type) {
      $info['files'][$file_id] = [
        'fid' => $file_id,
        'group_relationship_type' => $group_relationship_type,
        'status' => static::STATUS_PENDING,
      ];
    }

    $upload_id = $this->uuidGenerator->generate();
    $temp_store->set($upload_id, $info);

    return $upload_id;
  }

  /**
   * Set info for an upload.
   *
   * @param string $upload_id
   *   The upload identifier.
   * @param string $key
   *   The key for the data.
   * @param mixed $data
   *   The data to set.
   *
   * @return $this
   */
  public function setUploadInfo(string $upload_id, string $key, mixed $data): self {
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    $info = $this->getUploadInfo($upload_id);
    $info[$key] = $data;
    $temp_store->set($upload_id, $info);
    return $this;
  }

  /**
   * Clear the uploaded files from the temp store and delete them.
   *
   * @param string $upload_id
   *   The upload identifier.
   *
   * @return $this
   */
  public function clearUploadedFiles(string $upload_id) : self {
    $this->privateTempStoreFactory->get('group_media_bulk_upload')
      ->delete($upload_id);
    return $this;
  }

  /**
   * Set the status of a file in the bulk upload process.
   *
   * @param string $upload_id
   *   The upload identifier.
   * @param int $fid
   *   The file ID.
   * @param string $status
   *   The status to set.
   *
   * @return $this
   */
  public function setFileStatus(string $upload_id, int $fid, string $status): self {
    $info = $this->getUploadInfo($upload_id);

    if (empty($info['files'][$fid])) {
      return $this;
    }

    $info['files'][$fid]['status'] = $status;
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    $temp_store->set($upload_id, $info);

    return $this;
  }

  /**
   * Get the next file ID to be uploaded.
   *
   * @param string $upload_id
   *   The upload identifier.
   *
   * @return array|null
   *   The next file ID, or NULL if none found.
   */
  protected function getNextUpload(string $upload_id): ?array {
    $info = $this->getUploadInfo($upload_id);

    foreach ($info['files'] ?? [] as $item) {
      if (($item['status'] ?? NULL) == static::STATUS_PENDING) {
        return $item;
      }
    }

    return NULL;
  }

  /**
   * Get the next redirect URL for the bulk upload process.
   *
   * @param string $upload_id
   *   The upload identifier.
   *
   * @return \Drupal\Core\Url|null
   *   The redirect URL, or NULL if none found.
   */
  public function getNextBulkUploadRedirect(string $upload_id): ?Url {
    $info = $this->getUploadInfo($upload_id);

    $gid = $info['group'] ?? NULL;
    $next_upload = $this->getNextUpload($upload_id);

    if ($gid && $next_upload) {
      $relationship_type_id = $next_upload['group_relationship_type'];

      $relationship_type_storage = $this->entityTypeManager->getStorage('group_relationship_type');
      assert($relationship_type_storage instanceof GroupRelationshipTypeStorage);
      $relationship_type = $relationship_type_storage->load($relationship_type_id);
      assert($relationship_type instanceof GroupRelationshipTypeInterface);
      $plugin = $relationship_type->getPluginId();

      return Url::fromRoute('entity.group_relationship.create_form', [
        'group' => $gid,
        'plugin_id' => $plugin,
        'upload' => $upload_id,
      ]);
    }

    // Final redirect.
    return $this->getFinalDestination($upload_id);
  }

  /**
   * Gets the final redirect destination URL.
   */
  public function getFinalDestination(string $upload_id): ?Url {
    $info = $this->getUploadInfo($upload_id);

    if (!empty($info['redirect'])) {
      if ($info['redirect'] instanceof Url) {
        return $info['redirect'];
      }
      return Url::fromUri('internal:' . $info['redirect']);
    }

    return NULL;
  }

  /**
   * Obtain the bulk upload file from the 'upload' query parameter.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity or NULL available.
   */
  public function getBulkUploadFile(): ?FileInterface {
    if (!$upload_id = $this->getCurrentUploadId()) {
      return NULL;
    }

    $info = $this->getNextUpload($upload_id);
    if (!$file = $this->entityTypeManager->getStorage('file')->load($info['fid'])) {
      return NULL;
    }

    assert($file instanceof FileInterface);
    return $file;
  }

  /**
   * Get upload info from the temp store.
   *
   * @param string|null $upload_id
   *   The upload identifier. Optional, attempts to obtain the current upload
   *   from query param if not provided.
   *
   * @return array|null
   *   The upload info, or NULL if not available.
   */
  public function getUploadInfo(?string $upload_id = NULL): ?array {
    if (!$upload_id) {
      $upload_id = $this->getCurrentUploadId();
    }
    $temp_store = $this->privateTempStoreFactory->get('group_media_bulk_upload');
    return $temp_store->get($upload_id) ?? NULL;
  }

  /**
   * Get the current upload ID from the request.
   *
   * @return string|null
   *   The upload ID, or NULL if not available.
   */
  public function getCurrentUploadId(): ?string {
    return $this->requestStack->getCurrentRequest()->query->get('upload');
  }

}
