<?php

declare(strict_types=1);

namespace Drupal\group_media_bulk_upload;

use Drupal\Core\Utility\Token;
use Drupal\Component\Render\PlainTextOutput;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Mime\MimeTypeGuesserInterface;
use Symfony\Component\Validator\ConstraintViolationListInterface;
use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Environment;
use Drupal\file\Validation\FileValidatorInterface;
use Drupal\file\FileInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\media\MediaTypeInterface;

/**
 * Media manager.
 */
class BulkMediaManager {

  /**
   * Constructs a BulkMediaManager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\file\Validation\FileValidatorInterface $fileValidator
   *   The file validator.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system.
   * @param \Symfony\Component\Mime\MimeTypeGuesserInterface $mimeTypeGuesser
   *   The mime type guesser.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\group_media_bulk_upload\MediaTypeManager $mediaTypeManager
   *   The media type manager service.
   * @param \Drupal\group_media_bulk_upload\GroupMediaUploadAccess $uploadAccess
   *   Upload access service.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly FileValidatorInterface $fileValidator,
    protected readonly AccountInterface $currentUser,
    protected readonly FileSystemInterface $fileSystem,
    #[Autowire(service: 'file.mime_type.guesser')]
    protected readonly MimeTypeGuesserInterface $mimeTypeGuesser,
    protected readonly Token $token,
    protected readonly MediaTypeManager $mediaTypeManager,
    protected readonly GroupMediaUploadAccess $uploadAccess,
  ) {
    // No op.
  }

  /**
   * Gets the group media type for a given file URI.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group where the file is being uploaded.
   * @param string $uri
   *   The file URI.
   *
   * @return string|null
   *   The group content type ID, or NULL if none found.
   */
  public function getGroupMediaTypeForFile(GroupInterface $group, string $uri): ?string {
    foreach ($this->mediaTypeManager->getMediaTypes($group) as $type) {
      if (!$media_type = $this->mediaTypeManager->getMediaTypeForGroupRelationshipType($type)) {
        continue;
      }

      // Permission to create the media.
      if (!$this->uploadAccess->groupRelationshipTypeAccess($group, $type, $this->currentUser)->isAllowed()) {
        continue;
      }
      assert($media_type instanceof MediaTypeInterface);
      $extensions = $media_type->getSource()->getSourceFieldDefinition($media_type)->getSetting('file_extensions');
      if (!$extensions) {
        continue;
      }

      $regex = '/\.(' . preg_replace('/ +/', '|', preg_quote($extensions)) . ')$/i';
      if (preg_match($regex, $uri)) {
        return $type->id();
      }
    }

    return NULL;
  }

  /**
   * Validate a media upload.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file to validate.
   * @param string $type
   *   The group media type ID.
   *
   * @return \Symfony\Component\Validator\ConstraintViolationListInterface
   *   An array of error messages, or an empty array if valid.
   */
  public function validateMediaUpload(FileInterface $file, string $type): ConstraintViolationListInterface {
    $group_relationship_type = $this->entityTypeManager
      ->getStorage('group_relationship_type')
      ->load($type);
    assert($group_relationship_type instanceof GroupRelationTypeInterface);

    $media_type = $this->mediaTypeManager->getMediaTypeForGroupRelationshipType($group_relationship_type);
    $source = $media_type->getSource()->getSourceFieldDefinition($media_type);
    $field_settings = $source->getSettings();

    $upload_validators = [];

    $max_filesize = Bytes::toNumber(Environment::getUploadMaxSize());
    if (!empty($field_settings['max_filesize'])) {
      $max_filesize = min($max_filesize, Bytes::toNumber($field_settings['max_filesize']));
    }

    // There is always a file size limit due to the PHP server limit.
    $upload_validators['FileSizeLimit'] = ['fileLimit' => $max_filesize];

    if ($source->getType() == 'image') {
      $upload_validators['FileIsImage'] = [];
      if ($field_settings['max_resolution'] || $field_settings['min_resolution']) {
        $upload_validators['GroupMediaBulkUploadFileImageDimensions'] = [
          'maxDimensions' => $field_settings['max_resolution'],
          'minDimensions' => $field_settings['min_resolution'],
        ];
      }
    }

    return $this->fileValidator->validate($file, $upload_validators);
  }

  /**
   * Process an uploaded file.
   *
   * @param string $type
   *   The group media content type.
   * @param string $uri
   *   The URI to the uploaded file.
   */
  public function processUpload(string $type, string $uri): FileInterface {
    $group_relationship_type = $this->entityTypeManager
      ->getStorage('group_relationship_type')
      ->load($type);
    assert($group_relationship_type instanceof GroupRelationTypeInterface);

    // Copy the file to the destination directory per field settings.
    $media_type = $this->mediaTypeManager->getMediaTypeForGroupRelationshipType($group_relationship_type);
    $field = $media_type->getSource()->getSourceFieldDefinition($media_type);
    $settings = $field->getSettings();
    $destination = trim($settings['file_directory'], '/');
    $destination = PlainTextOutput::renderFromHtml($this->token->replace($destination));
    $destination_dir = $settings['uri_scheme'] . '://' . $destination;
    $this->fileSystem->prepareDirectory($destination_dir, FileSystemInterface::CREATE_DIRECTORY);
    $destination = $destination_dir . basename($uri);
    $uri = $this->fileSystem->move($uri, $destination);

    /** @var \Drupal\file\FileInterface $file */
    $file = $this->entityTypeManager->getStorage('file')->create([
      'uri' => $uri,
      'uid' => $this->currentUser->id(),
    ]);
    $file->setMimeType($this->mimeTypeGuesser->guessMimeType($uri));
    $file->setSize(@filesize($this->fileSystem->realPath($uri)));
    $file->setTemporary();
    return $file;
  }

}
