<?php

declare(strict_types=1);

namespace Drupal\group_media_bulk_upload\Form;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\group_media_bulk_upload\BulkMediaManager;
use Drupal\group_media_bulk_upload\UploadManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a group_media_bulk_upload form.
 */
class GroupMediaBulkUploadForm extends FormBase {

  /**
   * Constructs a new GroupMediaBulkUploadForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\file\FileRepositoryInterface $fileRepository
   *   The file repository.
   * @param \Drupal\group_media_bulk_upload\BulkMediaManager $bulkMediaManager
   *   The bulk media manager.
   * @param \Drupal\group_media_bulk_upload\UploadManager $uploadManager
   *   The upload manager.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly FileRepositoryInterface $fileRepository,
    protected readonly BulkMediaManager $bulkMediaManager,
    protected readonly UploadManager $uploadManager,
  ) {
    // No op.
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('file.repository'),
      $container->get('group_media_bulk_upload.bulk_media_manager'),
      $container->get('group_media_bulk_upload.upload_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'group_media_bulk_upload_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?GroupInterface $group = NULL): array {
    $form_state->set('group', $group);

    $form['upload_container']['dropzone'] = [
      '#type' => 'dropzonejs',
      '#dropzone_description' => $this->t('Drag files here to upload them'),
    ];

    $form['actions'] = [
      '#type' => 'actions',
      '#weight' => 100,
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Continue Upload'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Validation is optional.
    $file_ids = [];
    foreach ($form_state->getValue(['dropzone', 'uploaded_files']) as $upload) {
      // Determine media type for this file.
      if (!$type = $this->bulkMediaManager->getGroupMediaTypeForFile($form_state->get('group'), $upload['path'])) {
        $this->messenger()->addError($this->t('Invalid extension for file %file.', ['%file' => $upload['filename']]));
        continue;
      }

      $file = $this->bulkMediaManager->processUpload($type, $upload['path']);

      $errors = $this->bulkMediaManager->validateMediaUpload($file, $type);
      if (count($errors) > 0) {
        $message = [];
        foreach ($errors as $error) {
          $message[] = $error->getMessage();
        }
        $this->messenger()->addError($this->t('Unable to upload %file: @message', [
          '%file' => $upload['filename'],
          '@message' => new FormattableMarkup(implode(' ', $message), []),
        ]));
        continue;
      }

      $file->save();
      $file_ids[$file->id()] = $type;
    }

    $form_state->set('file_ids', $file_ids);

    if (empty($file_ids)) {
      // No valid files uploaded.
      $form_state->setErrorByName('dropzone', $this->t('No valid files were uploaded.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $upload_id = $this->uploadManager->initializeUpload($form_state->get('group'), $form_state->get('file_ids'), $this->getRedirectDestination()->get());
    $form_state->set('group_media_bulk_upload_id', $upload_id);

    if ($redirect = $this->uploadManager->getNextBulkUploadRedirect($upload_id)) {
      $form_state->setRedirectUrl($redirect);
    }
  }

}
