<?php

declare(strict_types=1);

namespace Drupal\group_media_bulk_upload\Hook;

use Drupal\Core\Entity\Display\EntityFormDisplayInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\group_media_bulk_upload\UploadManager;
use Drupal\media\MediaInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\media\MediaForm;

/**
 * Hooks for Group Media Bulk Upload module.
 */
class GroupMediaBulkUploadHooks {

  use StringTranslationTrait;

  /**
   * Constructs a GroupMediaBulkUploadHooks.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\group_media_bulk_upload\UploadManager $uploadManager
   *   The upload manager service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   */
  public function __construct(
    protected readonly RequestStack $requestStack,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly AccountInterface $currentUser,
    protected readonly UploadManager $uploadManager,
    protected readonly MessengerInterface $messenger,
    protected readonly EntityTypeBundleInfoInterface $entityTypeBundleInfo,
  ) {
    // No op.
  }

  /**
   * Add Views data definition.
   */
  #[Hook('views_data_alter')]
  public function viewsDataAlter(array &$data): void {
    $data['views']['group_media_bulk_upload'] = [
      'title' => $this->t('Group Media Bulk Upload'),
      'help' => $this->t('Provides a form for bulk media upload within a group.'),
      'area' => [
        'id' => 'group_media_bulk_upload',
      ],
    ];
  }

  /**
   * Sets the media source field value for the bulk upload.
   */
  #[Hook('media_prepare_form')]
  public function prepareMediaForm(MediaInterface $media, $operation, FormStateInterface $form_state): void {
    if ($operation != 'add') {
      return;
    }

    $form_object = $form_state->getFormObject();
    if (!$form_object instanceof MediaForm) {
      return;
    }

    if (!$file = $this->uploadManager->getBulkUploadFile()) {
      return;
    }

    if ($file->getOwnerId() != $this->currentUser->id()) {
      return;
    }

    $source_field = $media->getSource()->getConfiguration()['source_field'];
    if (!$media->{$source_field}->isEmpty()) {
      return;
    }

    $media->{$source_field}->setValue($file->id());
  }

  /**
   * Alter the media form to add bulk upload navigation buttons.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  #[Hook('form_alter')]
  public function alterMediaForm(array &$form, FormStateInterface $form_state): void {
    $form_object = $form_state->getFormObject();
    if (!$form_object instanceof MediaForm) {
      return;
    }

    if (!$this->uploadManager->getBulkUploadFile()) {
      return;
    }

    $form_state->set('group_media_bulk_upload_id', $this->uploadManager->getCurrentUploadId());

    // Adjust the submit button.
    $form['actions']['submit']['#value'] = $this->t('Continue');
    $form['actions']['submit']['#submit'][] = [$this, 'mediaFormSubmit'];

    $media_type = $this->entityTypeManager->getStorage('media_type')
      ->load($form_object->getEntity()->bundle());

    // Add skip button.
    $form['actions']['skip'] = [
      '#value' => $this->t('Skip @type', ['@type' => $media_type->label()]),
      '#type' => 'submit',
      '#submit' => [[$this, 'mediaFormSkip']],
      '#attributes' => ['formnovalidate' => 'formnovalidate'],
      '#limit_validation_errors' => [],
    ];

    // Add cancel button.
    $form['actions']['cancel'] = [
      '#value' => $this->t('Cancel all'),
      '#type' => 'submit',
      '#submit' => [[$this, 'mediaFormCancel']],
      '#attributes' => ['formnovalidate' => 'formnovalidate'] ,
      '#limit_validation_errors' => [],
    ];

    // Add the bulk upload progress indicator.
    if (($display = $form_state->get('form_display')) instanceof EntityFormDisplayInterface) {
      if ($component = $display->getComponent('group_media_bulk_upload_progress')) {
        $form['group_media_bulk_upload_progress'] = [
          '#theme' => 'group_media_bulk_upload_progress',
          '#weight' => $component['weight'] ?? 0,
        ];
      }
    }
  }

  /**
   * Submit callback to redirect to the next upload or final destination.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  public function mediaFormRedirect(array $form, FormStateInterface $form_state): void {
    if (!$upload_id = $form_state->get('group_media_bulk_upload_id')) {
      return;
    }

    $file = $this->uploadManager->getBulkUploadFile();
    $this->uploadManager->setFileStatus($upload_id, (int) $file->id(), UploadManager::STATUS_COMPLETE);

    if ($redirect = $this->uploadManager->getNextBulkUploadRedirect($upload_id)) {
      $form_state->setRedirectUrl($redirect);
    }
  }

  /**
   * Submit callback for skipping a media upload.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  public function mediaFormSkip(array $form, FormStateInterface $form_state): void {
    if (!$upload_id = $form_state->get('group_media_bulk_upload_id')) {
      return;
    }

    $file = $this->uploadManager->getBulkUploadFile();
    $this->uploadManager->setFileStatus($upload_id, (int) $file->id(), UploadManager::STATUS_CANCELLED);

    $this->messenger->addWarning($this->t('Uploaded item skipped.'));

    if ($redirect = $this->uploadManager->getNextBulkUploadRedirect($upload_id)) {
      $form_state->setRedirectUrl($redirect);
    }
  }

  /**
   * Submit callback for canceling the bulk media upload.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  public function mediaFormCancel(array $form, FormStateInterface $form_state): void {
    if (!$upload_id = $form_state->get('group_media_bulk_upload_id')) {
      return;
    }

    if ($redirect = $this->uploadManager->getFinalDestination($upload_id)) {
      $form_state->setRedirectUrl($redirect);
    }

    $this->uploadManager->clearUploadedFiles($upload_id);
    $this->messenger->addWarning($this->t('Bulk upload canceled.'));
  }

  /**
   * Submit callback for continue/save.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  public function mediaFormSubmit(array $form, FormStateInterface $form_state): void {
    if (!$upload_id = $form_state->get('group_media_bulk_upload_id')) {
      return;
    }

    $file = $this->uploadManager->getBulkUploadFile();
    $this->uploadManager->setFileStatus($upload_id, (int) $file->id(), UploadManager::STATUS_COMPLETE);

    if ($redirect = $this->uploadManager->getNextBulkUploadRedirect($upload_id)) {
      $form_state->setRedirectUrl($redirect);
    }
  }

  /**
   * Sets extra field info for bulk upload of media entities.
   *
   * @return array
   *   The extra field info array.
   */
  #[Hook('entity_extra_field_info')]
  public function entityExtraFieldInfo(): array {
    $extra = [];

    foreach ($this->entityTypeBundleInfo->getBundleInfo('media') as $bundle => $bundle_info) {
      $extra['media'][$bundle]['form']['group_media_bulk_upload_progress'] = [
        'label' => $this->t('Bulk Upload Progress'),
        'description' => $this->t('Provides a progress indicator for bulk uploads.'),
        'weight' => -50,
        'visible' => TRUE,
      ];
    }

    return $extra;
  }

  /**
   * Set theme hooks.
   */
  #[Hook('theme')]
  public function theme() {
    return [
      'group_media_bulk_upload_progress' => [
        'render element' => 'progress',
      ],
    ];
  }

}
