<?php

declare(strict_types=1);

namespace Drupal\group_media_bulk_upload\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\group_media_bulk_upload\BulkMediaManager;
use Drupal\media\MediaInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\media\MediaForm;

/**
 * Hooks for Group Media Bulk Upload module.
 */
class GroupMediaBulkUploadHooks {

  use StringTranslationTrait;

  /**
   * Constructs a GroupMediaBulkUploadHooks.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\group_media_bulk_upload\BulkMediaManager $bulkMediaManager
   *   The bulk media manager service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    protected readonly RequestStack $requestStack,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly AccountInterface $currentUser,
    protected readonly BulkMediaManager $bulkMediaManager,
    protected readonly MessengerInterface $messenger,
  ) {
    // No op.
  }

  /**
   * Add Views data definition.
   */
  #[Hook('views_data_alter')]
  public function viewsDataAlter(array &$data): void {
    $data['views']['group_media_bulk_upload'] = [
      'title' => $this->t('Group Media Bulk Upload'),
      'help' => $this->t('Provides a form for bulk media upload within a group.'),
      'area' => [
        'id' => 'group_media_bulk_upload',
      ],
    ];
  }

  /**
   * Sets the media source field value from the 'bulk_fid' query parameter.
   */
  #[Hook('media_prepare_form')]
  public function prepareMediaForm(MediaInterface $media, $operation, FormStateInterface $form_state): void {
    if ($operation != 'add') {
      return;
    }

    $form_object = $form_state->getFormObject();
    if (!$form_object instanceof MediaForm) {
      return;
    }

    if (!$file = $this->bulkMediaManager->getBulkUploadFile()) {
      return;
    }

    if ($file->getOwnerId() != $this->currentUser->id()) {
      return;
    }

    $source_field = $media->getSource()->getConfiguration()['source_field'];
    if (!$media->{$source_field}->isEmpty()) {
      return;
    }

    $media->{$source_field}->setValue($file->id());
  }

  /**
   * Alter the media form to add bulk upload navigation buttons.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  #[Hook('form_alter')]
  public function alterMediaForm(array &$form, FormStateInterface $form_state): void {
    $form_object = $form_state->getFormObject();
    if (!$form_object instanceof MediaForm) {
      return;
    }

    if (!$this->bulkMediaManager->getBulkUploadFile()) {
      return;
    }

    $form['actions']['submit']['#value'] = $this->t('Continue');
    $form['actions']['submit']['#submit'][] = [$this, 'mediaFormRedirect'];

    $form['actions']['skip'] = [
      '#value' => $this->t('Skip'),
      '#type' => 'submit',
      '#submit' => [
        [$this, 'mediaFormRedirect'],
        [$this, 'mediaFormSkip'],
      ],
      '#attributes' => ['formnovalidate' => 'formnovalidate'],
      '#limit_validation_errors' => [],
    ];

    $form['actions']['cancel'] = [
      '#value' => $this->t('Cancel'),
      '#type' => 'submit',
      '#submit' => [
        [$this, 'mediaFormRedirect'],
        [$this, 'mediaFormCancel'],
      ],
      '#attributes' => ['formnovalidate' => 'formnovalidate'] ,
      '#limit_validation_errors' => [],
    ];
  }

  /**
   * Submit callback to redirect to the next upload or final destination.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  public function mediaFormRedirect(array $form, FormStateInterface $form_state): void {
    if ($redirect = $this->bulkMediaManager->nextUploadedFile()->getNextBulkUploadRedirect()) {
      $form_state->setRedirectUrl($redirect);
    }
  }

  /**
   * Submit callback for skipping a media upload.
   */
  public function mediaFormSkip(): void {
    $this->messenger->addWarning($this->t('Uploaded item skipped.'));
  }

  /**
   * Submit callback for canceling the bulk media upload.
   */
  public function mediaFormCancel(): void {
    $this->bulkMediaManager->clearUploadedFiles();
    $this->messenger->addWarning($this->t('Bulk upload canceled.'));
  }

}
