<?php

declare(strict_types=1);

namespace Drupal\Tests\group_notify\Functional;

use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Test\AssertMailTrait;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\group_notify\Traits\GroupCreationTrait;
use Drupal\Tests\group_notify\Traits\TestGroupsTrait;
use Drupal\group\PermissionScopeInterface;

/**
 * Test basic functionality of group_notify.
 *
 * @group group_notify
 */
class GroupNotifyTest extends BrowserTestBase {

  use AssertMailTrait;
  use GroupCreationTrait;
  use TestGroupsTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'group_notify',
    'group',
    'gnode',
    'node',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Authenticated administrator user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * Regular authenticated user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser1;

  /**
   * Another regular authenticated user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser2;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create node type.
    $this->drupalCreateContentType([
      'type' => 'article',
      'name' => 'Article',
    ]);

    $this->adminUser = $this->drupalCreateUser([
      'access administration pages',
      'access group overview',
      'administer account settings',
      'administer content types',
      'administer group',
      'administer users',
      'bypass node access',
    ]);
    $this->testUser1 = $this->drupalCreateUser([
      'access content',
    ]);
    $this->testUser2 = $this->drupalCreateUser([
      'access content',
    ]);

    $this->drupalLogin($this->adminUser);

    // Setup the group types and test groups from the TestGroupsTrait.
    $this->initializeTestGroups();

    // Enable article nodes to be assigned to only 'A' group type.
    $this->entityTypeManager->getStorage(group_notify_get_group_relationship_type_id())
      ->createFromPlugin($this->groupTypeA, 'group_node:article')->save();

    // Create admin role for group types.
    $this->entityTypeManager->getStorage('group_role')
      ->create([
        'id' => $this->randomMachineName(8),
        'label' => $this->randomString(),
        'group_type' => $this->groupTypeA->id(),
        'scope' => PermissionScopeInterface::INSIDER_ID,
        'global_role' => $this->adminUser->getRoles(TRUE)[0],
        'admin' => TRUE,
      ])->save();

    // Let regular group members view and add content to the groups.
    $this->entityTypeManager->getStorage('group_role')
      ->create([
        'id' => $this->randomMachineName(8),
        'label' => $this->randomString(),
        'group_type' => $this->groupTypeA->id(),
        'scope' => PermissionScopeInterface::INSIDER_ID,
        'global_role' => AccountInterface::AUTHENTICATED_ROLE,
        'permissions' => [
          'view group',
          'create group_node:article relationship',
          'delete own group_node:article relationship',
          'update own group_node:article relationship',
          'view group_node:article relationship',
          'view group_node:article entity',
        ],
      ])->save();

    // Subscribe the admin user to both A1 and A2.
    $this->groupA1->addMember($this->adminUser);
    $this->groupA2->addMember($this->adminUser);

    // Subscribe the test1 user to both A1 and A2.
    $this->groupA1->addMember($this->testUser1);
    $this->groupA2->addMember($this->testUser1);

    // Subscribe the test2 user to only A1.
    $this->groupA1->addMember($this->testUser2);
  }

  /**
   * Tests basic functionality of group_notify.
   */
  public function testGroupNotify() {
    // Should have no emails at the start.
    $this->assertCount(0, $this->getMails());

    // Create an article, outside of any groups.
    $this->drupalGet('/node/add/article');
    $page = $this->getSession()->getPage();
    $new_title = $this->randomString();
    $title = $page->findField('title[0][value]');
    $title->setValue($new_title);
    $page->findButton('Save')->click();
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains("Article $new_title has been created.");

    // Still should be no emails, since this article is not part of a group.
    $this->assertCount(0, $this->getMails());

    // Ensure the group memberships are what we expect.
    $groupA1_members = $this->groupA1->getMembers();
    $this->assertCount(3, $groupA1_members);
    $this->assertEquals($this->adminUser->id(), $groupA1_members[0]->getUser()->id());
    $this->assertEquals($this->testUser1->id(), $groupA1_members[1]->getUser()->id());
    $this->assertEquals($this->testUser2->id(), $groupA1_members[2]->getUser()->id());

    $groupA2_members = $this->groupA2->getMembers();
    $this->assertCount(2, $groupA2_members);
    $this->assertEquals($this->adminUser->id(), $groupA2_members[0]->getUser()->id());
    $this->assertEquals($this->testUser1->id(), $groupA2_members[1]->getUser()->id());

    // Now, add an article inside group A1.
    $this->drupalGet('/group/' . $this->groupA1->id() . '/content/create/group_node:article');
    $this->assertSession()->statusCodeEquals(200);
    $page = $this->getSession()->getPage();
    $new_title = $this->randomString();
    $title = $page->findField('title[0][value]');
    $title->setValue($new_title);
    $page->findButton('Save')->click();
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains("Article $new_title has been created.");

    // Still should be no emails, since we haven't configured group_notify yet.
    $this->assertCount(0, $this->getMails());

    // Configure group_notify on A-Group article nodes.
    $this->drupalGet('/admin/group/content/manage/' . $this->groupTypeA->id() . '-group_node-article');
    $this->assertSession()->statusCodeEquals(200);
    // Should default to off.
    $this->assertSession()->checkboxNotChecked('edit-notify');
    $this->submitForm(['edit-notify' => TRUE], 'Save configuration');
    // This sends us back to the content overview page.
    $this->assertSession()->statusCodeEquals(200);
    // Fetch the configuration form, and group_notify should now be enabled.
    $this->drupalGet('/admin/group/content/manage/' . $this->groupTypeA->id() . '-group_node-article');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->checkboxChecked('edit-notify');

    // Now, add another article to Group A1, should get 2 emails.
    $this->drupalGet('/group/' . $this->groupA1->id() . '/content/create/group_node:article');
    $this->assertSession()->statusCodeEquals(200);
    $page = $this->getSession()->getPage();
    $new_title = $this->randomString();
    $title = $page->findField('title[0][value]');
    $title->setValue($new_title);
    $page->findButton('Save')->click();
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains("Article $new_title has been created.");

    // Should have 2 emails, one for each test user (but not the admin, since
    // they are the node author).
    $test_mails = $this->getMails();
    $this->assertCount(2, $test_mails);

    $this->assertEquals('group_notify', $test_mails[0]['module']);
    $this->assertEquals('group_post', $test_mails[0]['key']);
    $this->assertEquals('group_notify_group_post', $test_mails[0]['id']);
    $this->assertEquals('New group content: ' . $new_title, $test_mails[0]['subject']);
    $this->assertEquals($this->testUser1->getEmail(), $test_mails[0]['to']);

    $this->assertEquals('group_notify', $test_mails[1]['module']);
    $this->assertEquals('group_post', $test_mails[1]['key']);
    $this->assertEquals('group_notify_group_post', $test_mails[1]['id']);
    $this->assertEquals('New group content: ' . $new_title, $test_mails[1]['subject']);
    $this->assertEquals($this->testUser2->getEmail(), $test_mails[1]['to']);

    // Now create an article in groupA2.
    $this->drupalGet('/group/' . $this->groupA2->id() . '/content/create/group_node:article');
    $this->assertSession()->statusCodeEquals(200);
    $page = $this->getSession()->getPage();
    $new_title = $this->randomString();
    $title = $page->findField('title[0][value]');
    $title->setValue($new_title);
    // Save the article.
    $page->findButton('Save')->click();
    // Confirm that saving created the new article.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains("Article $new_title has been created.");

    // Should have 1 new email (3 total).
    $test_mails = $this->getMails();
    $this->assertCount(3, $test_mails);
    // Make sure the last email was sent to testUser1 with the new subject.
    $this->assertEquals('group_notify', $test_mails[2]['module']);
    $this->assertEquals('group_post', $test_mails[2]['key']);
    $this->assertEquals('group_notify_group_post', $test_mails[2]['id']);
    $this->assertEquals('New group content: ' . $new_title, $test_mails[2]['subject']);
    $this->assertEquals($this->testUser1->getEmail(), $test_mails[2]['to']);
  }

}
