<?php

namespace Drupal\group_purl\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\path_alias\AliasManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\group_purl\Context\GroupPurlContext;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Class GroupViewController.
 */
class GroupViewController extends ControllerBase {

  /**
   * AliasManager definition.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;

  /**
   * Drupal\group_purl\Context\GroupPurlContext definition.
   *
   * @var \Drupal\group_purl\Context\GroupPurlContext
   */
  protected $groupPurlContextProvider;

  /**
   * Constructs a new GroupViewController object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, AliasManagerInterface $aliasManager, GroupPurlContext $group_purl_context_provider) {
    $this->entityTypeManager = $entity_type_manager;
    $this->aliasManager = $aliasManager;
    $this->groupPurlContextProvider = $group_purl_context_provider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('path_alias.manager'),
      $container->get('group_purl.context_provider'),
    );
  }

  /**
   * Override view to check group context and redirect if necessary.
   *
   * @param \Drupal\Core\Entity\EntityInterface $group
   *   The group entity to view.
   * @param string $view_mode
   *   The view mode to use.
   * @param string $langcode
   *   The language code to use.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   The view render array or a redirect response.
   */
  public function view(EntityInterface $group, $view_mode = 'full', $langcode = NULL) {
    // Check if we're in a PURL subrequest by looking at request attributes.
    $request = \Drupal::request();
    $purl_modifiers = $request->attributes->get('purl.matched_modifiers', []);

    \Drupal::logger('group_purl')->debug('GroupViewController: Group @gid, PURL modifiers in request: @count', [
      '@gid' => $group->id(),
      '@count' => count($purl_modifiers),
    ]);

    // If we have PURL modifiers, we're in a subrequest with group context.
    if (!empty($purl_modifiers)) {
      foreach ($purl_modifiers as $match) {
        if (isset($match['method']) && $match['method']->getPluginId() === 'group_prefix') {
          $current_group_id = $match['value'];
          \Drupal::logger('group_purl')->debug('GroupViewController: In group context subrequest for group @gid', ['@gid' => $current_group_id]);

          // If we're accessing a different group than current context, redirect.
          if ($current_group_id != $group->id()) {
            $target_group_alias = $this->aliasManager->getAliasByPath('/group/' . $group->id());
            $target_group_modifier = ltrim($target_group_alias, '/');

            // If target group has no meaningful alias, exit PURL context.
            if ($target_group_alias === '/group/' . $group->id() || $target_group_modifier === 'group/' . $group->id()) {
              $redirect_url = '/group/' . $group->id();
              \Drupal::logger('group_purl')->debug('GroupViewController: Target group has no alias, redirecting to @url', ['@url' => $redirect_url]);
              return new RedirectResponse($redirect_url);
            }

            // Redirect to correct group context.
            $redirect_url = '/' . $target_group_modifier;
            \Drupal::logger('group_purl')->debug('GroupViewController: Wrong group context (@current vs @correct), redirecting to @url', [
              '@current' => $current_group_id,
              '@correct' => $group->id(),
              '@url' => $redirect_url,
            ]);
            return new RedirectResponse($redirect_url);
          }

          // Correct group context, proceed normally.
          \Drupal::logger('group_purl')->debug('GroupViewController: Correct group context, proceeding normally');
          return $this->entityTypeManager->getViewBuilder('group')->view($group, $view_mode, $langcode);
        }
      }
    }

    // No PURL context - check if this group should have context.
    $group_alias = $this->aliasManager->getAliasByPath('/group/' . $group->id());
    $group_modifier = ltrim($group_alias, '/');

    // If group has a meaningful alias, redirect to group context.
    if ($group_alias !== '/group/' . $group->id() && $group_modifier !== 'group/' . $group->id()) {
      $redirect_url = '/' . $group_modifier;
      \Drupal::logger('group_purl')->debug('GroupViewController: Group has alias, redirecting to @url', ['@url' => $redirect_url]);
      return new RedirectResponse($redirect_url);
    }

    // Group has no meaningful alias, render normally.
    \Drupal::logger('group_purl')->debug('GroupViewController: Group has no meaningful alias, rendering normally');
    return $this->entityTypeManager->getViewBuilder('group')->view($group, $view_mode, $langcode);
  }

}
