<?php

namespace Drupal\group_purl\Controller;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\group\Entity\GroupRelationship;
use Drupal\node\Controller\NodeViewController;
use Drupal\path_alias\AliasManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\group_purl\Context\GroupPurlContext;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Class NodeGroupViewController.
 */
class NodeGroupViewController extends NodeViewController {

  /**
   * AliasManger definition.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;
  /**
   * Drupal\group_purl\Context\GroupPurlContext definition.
   *
   * @var \Drupal\group_purl\Context\GroupPurlContext
   */
  protected $groupPurlContextProvider;

  /**
   * Constructs a new NodeGroupViewController object.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, RendererInterface $renderer, GroupPurlContext $group_purl_context_provider, AliasManagerInterface $aliasManager, EntityRepositoryInterface $entity_repository, ?AccountInterface $current_user = NULL) {
    parent::__construct($entityTypeManager, $renderer, $current_user, $entity_repository);
    $this->aliasManager = $aliasManager;
    $this->groupPurlContextProvider = $group_purl_context_provider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('renderer'),
      $container->get('group_purl.context_provider'),
      $container->get('path_alias.manager'),
      $container->get('entity.repository'),
      $container->get('current_user'),
    );
  }

  /**
   * Override view to check group context, and group membership.
   * If this node should appear in a different group context, redirect to it.
   *
   * {@inheritdoc}
   */
  public function view(EntityInterface $node, $view_mode = 'full', $langcode = NULL) {
    // Check if we're in a PURL subrequest by looking at request attributes.
    $request = \Drupal::request();
    $purl_modifiers = $request->attributes->get('purl.matched_modifiers', []);
    $request_path = $request->getPathInfo();

    // FIRST: Check if we have a nested/bad URL by detecting multiple group contexts in the path.
    if ($this->hasNestedGroupContexts($request_path)) {
      \Drupal::logger('group_purl')->error('NodeGroupViewController: Detected nested group contexts in path @path, redirecting to correct entity URL', [
        '@path' => $request_path,
      ]);

      // Get the correct URL from the entity and redirect to it.
      $correct_url = $node->toUrl()->toString();
      return new RedirectResponse($correct_url);
    }

    // If we have PURL modifiers, we're in a subrequest with group context
    // But we need to verify it's the CORRECT group context.
    if (!empty($purl_modifiers)) {
      foreach ($purl_modifiers as $match) {
        if (isset($match['method']) && $match['method']->getPluginId() === 'group_prefix') {
          $current_group_id = $match['value'];
          \Drupal::logger('group_purl')->debug('NodeGroupViewController: In group context subrequest for group @gid', ['@gid' => $current_group_id]);

          // Check if node belongs to this group.
          $group_contents = GroupRelationship::loadByEntity($node);
          if (!empty($group_contents)) {
            foreach ($group_contents as $group_content) {
              if ($group_content->getGroup()->id() == $current_group_id) {
                \Drupal::logger('group_purl')->debug('NodeGroupViewController: Node belongs to current group context, proceeding normally');
                return parent::view($node, $view_mode, $langcode);
              }
            }

            // Node doesn't belong to current group - redirect to correct group.
            $group_content = reset($group_contents);
            $group = $group_content->getGroup();
            $group_alias = $this->aliasManager->getAliasByPath('/group/' . $group->id());
            $group_modifier = ltrim($group_alias, '/');
            $node_alias = $this->aliasManager->getAliasByPath('/node/' . $node->id());
            $redirect_url = '/' . $group_modifier . $node_alias;

            \Drupal::logger('group_purl')->debug('NodeGroupViewController: Wrong group context (@current vs @correct), redirecting to @url', [
              '@current' => $current_group_id,
              '@correct' => $group->id(),
              '@url' => $redirect_url,
            ]);
            return new RedirectResponse($redirect_url);
          }

          // Node not in any group but we're in group context - redirect to remove group context.
          $node_alias = $this->aliasManager->getAliasByPath('/node/' . $node->id());
          \Drupal::logger('group_purl')->debug('NodeGroupViewController: Node not in group but in group context, redirecting to @url', ['@url' => $node_alias]);
          return new RedirectResponse($node_alias);
        }
      }
    }

    // Check if this node type should keep context and is in a group.
    $node_type = \Drupal::entityTypeManager()->getStorage('node_type')->load($node->bundle());
    $purl_settings = $node_type->getThirdPartySettings('purl');
    $keep_context = isset($purl_settings['keep_context']) && $purl_settings['keep_context'];

    \Drupal::logger('group_purl')->debug('NodeGroupViewController: Node type @type keep_context: @keep', [
      '@type' => $node->bundle(),
      '@keep' => $keep_context ? 'TRUE' : 'FALSE',
    ]);

    // Only redirect if keep_context is TRUE and node is in a group.
    if ($keep_context) {
      $group_contents = GroupRelationship::loadByEntity($node);
      if (!empty($group_contents)) {
        $group_content = reset($group_contents);
        $group = $group_content->getGroup();
        $group_alias = $this->aliasManager->getAliasByPath('/group/' . $group->id());
        $group_modifier = ltrim($group_alias, '/');
        $node_alias = $this->aliasManager->getAliasByPath('/node/' . $node->id());
        $redirect_url = '/' . $group_modifier . $node_alias;

        \Drupal::logger('group_purl')->debug('NodeGroupViewController: Node should keep context and is in group, redirecting to @url', ['@url' => $redirect_url]);
        return new RedirectResponse($redirect_url);
      }
    }

    // Either keep_context is FALSE or node is not in a group - render normally.
    \Drupal::logger('group_purl')->debug('NodeGroupViewController: Rendering normally (keep_context=@keep or not in group)', ['@keep' => $keep_context ? 'TRUE' : 'FALSE']);
    return parent::view($node, $view_mode, $langcode);
  }

  /**
   * Detects if a path has nested group contexts (e.g., /tourism/business/development-plan).
   *
   * @param string $path
   *   The request path to check.
   *
   * @return bool
   *   TRUE if nested group contexts are detected.
   */
  protected function hasNestedGroupContexts($path) {
    // Get all group aliases to detect nested contexts.
    $group_aliases = $this->getAllGroupAliases();

    if (empty($group_aliases)) {
      return FALSE;
    }

    // Check if path contains multiple group aliases.
    $path_parts = explode('/', trim($path, '/'));
    $found_contexts = 0;

    foreach ($path_parts as $part) {
      if (in_array($part, $group_aliases)) {
        $found_contexts++;
        if ($found_contexts > 1) {
          // Found nested contexts.
          return TRUE;
        }
      }
    }

    return FALSE;
  }

  /**
   * Gets all group aliases for nested context detection.
   */
  protected function getAllGroupAliases() {
    static $aliases = NULL;

    if ($aliases === NULL) {
      $aliases = [];

      try {
        $groups = $this->entityTypeManager->getStorage('group')->loadMultiple();
        foreach ($groups as $group) {
          $group_alias = $this->aliasManager->getAliasByPath('/group/' . $group->id());
          $group_modifier = ltrim($group_alias, '/');

          // Only include meaningful aliases (not system paths)
          if ($group_alias !== '/group/' . $group->id() && $group_modifier !== 'group/' . $group->id()) {
            $aliases[] = $group_modifier;
          }
        }
      }
      catch (\Exception $e) {
        \Drupal::logger('group_purl')->error('NodeGroupViewController: Error loading group aliases: @message', [
          '@message' => $e->getMessage(),
        ]);
      }
    }

    return $aliases;
  }

}
