<?php

declare(strict_types=1);

namespace Drupal\group_purl\Hook;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Site\Settings;
use Drupal\group\Entity\GroupInterface;
use Drupal\group_purl\Plugin\Purl\Provider\GroupPurlProvider;

/**
 * Hook implementations for group_purl.
 */
class GroupPurlHooks {

  /**
   * Implements hook_entity_insert().
   *
   * Add entities created in a group context to the group automatically.
   */
  #[Hook('entity_insert')]
  public function entityInsert(EntityInterface $entity): void {
    // Check if current route is for adding group relationships manually.
    $route_match = \Drupal::routeMatch();
    if ($route_match->getRouteName() && str_starts_with($route_match->getRouteName(), 'entity.group_relationship')) {
      return;
    }

    $context = \Drupal::service('group_purl.context_provider');
    $contexts = $context->getRuntimeContexts(['group']);
    if (!empty($contexts['group'])) {
      $group = $contexts['group']->getContextValue();

      // Check if this entity type has an available group relationship plugin.
      $plugin_id = $this->getAvailableGroupRelationshipPlugin($entity, $group);
      if ($plugin_id) {
        // Use static cache to prevent duplicate relationships in the same request.
        $cache = &drupal_static(__FUNCTION__, []);
        $cache_key = $entity->id() . ':' . $group->id() . ':' . $plugin_id;

        if (!isset($cache[$cache_key])) {
          $group->addRelationship($entity, $plugin_id);
          $cache[$cache_key] = TRUE;
        }
      }
    }
  }

  /**
   * Get an available group relationship plugin ID for an entity and group.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to find a plugin for.
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group to check available plugins.
   *
   * @return string|null
   *   The plugin ID if available, NULL otherwise.
   */
  private function getAvailableGroupRelationshipPlugin(EntityInterface $entity, $group): ?string {
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();

    // Get the group relation type manager to find available plugins.
    $relation_manager = \Drupal::service('group_relation_type.manager');
    $available_plugins = $relation_manager->getPluginIdsByEntityTypeId($entity_type_id);

    foreach ($available_plugins as $plugin_id) {
      // Check if this group type has this plugin installed.
      if ($group->getGroupType()->hasPlugin($plugin_id)) {
        // For plugins that are entity type + bundle specific, verify bundle matches.
        if (str_contains($plugin_id, ':')) {
          $plugin_bundle = substr($plugin_id, strpos($plugin_id, ':') + 1);
          if ($plugin_bundle === $bundle) {
            return $plugin_id;
          }
        } else {
          // For entity type only plugins (like group_membership), return it.
          return $plugin_id;
        }
      }
    }

    return NULL;
  }

  /**
   * Implements hook_link_alter().
   *
   * Alter links to work with both subdomain and path prefix PURL methods.
   */
  #[Hook('link_alter')]
  public function linkAlter(&$variables): void {
    $options = &$variables['options'];

    if (!isset($options['purl_context']) || $options['absolute']) {
      return;
    }

    // Get the PURL provider configuration to determine method type.
    $purl_providers = \Drupal::entityTypeManager()
      ->getStorage('purl_provider')
      ->loadMultiple();

    $group_provider = NULL;
    foreach ($purl_providers as $provider) {
      if ($provider->getProviderKey() === 'group_purl_provider') {
        $group_provider = $provider;
        break;
      }
    }

    if (!$group_provider) {
      return;
    }

    $method_key = $group_provider->getMethodKey();

    // Handle different PURL methods.
    switch ($method_key) {
      case 'subdomain':
        $this->handleSubdomainMethod($options);
        break;

      case 'path_prefix':
        $this->handlePathPrefixMethod($options);
        break;
    }
  }

  /**
   * Handle subdomain method for link altering.
   */
  private function handleSubdomainMethod(array &$options): void {
    $base_domain = Settings::get('purl_base_domain');
    if (!$base_domain || !isset($options['purl_context']['modifier'])) {
      return;
    }

    $linkhost = $options['purl_context']['modifier'] . '.' . $base_domain;
    $host = \Drupal::service('router.request_context')->getHost();

    if ($linkhost != $host) {
      $options['host'] = $linkhost;
      $options['absolute'] = TRUE;
    }
  }

  /**
   * Handle path prefix method for link altering.
   */
  private function handlePathPrefixMethod(array &$options): void {
    if (isset($options['purl_context']['modifier'])) {
      // For path prefix, we let PURL's own processors handle the URL modification
      // We just need to ensure the context is properly set
      // The actual path modification will be handled by PURL's URL generator.
    }
  }

  /**
   * Implements hook_group_insert().
   */
  #[Hook('group_insert')]
  public function groupInsert(GroupInterface $group): void {
    Cache::invalidateTags([GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY]);
  }

  /**
   * Implements hook_group_delete().
   */
  #[Hook('group_delete')]
  public function groupDelete(GroupInterface $group): void {
    Cache::invalidateTags([GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY]);
  }

}
