<?php

namespace Drupal\group_purl\Plugin\Purl\Method;

use Drupal\purl\Plugin\Purl\Method\PathPrefixMethod;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Routing\TrustedRedirectResponse;

/**
 * @PurlMethod(
 *   id="group_prefix",
 *   title = @Translation("Group Content."),
 *   stages = {
 *     Drupal\purl\Plugin\Purl\Method\MethodInterface::STAGE_PROCESS_OUTBOUND
 *   }
 * )
 */
class GroupPrefixMethod extends PathPrefixMethod {

  /**
   *
   */
  public function contains(Request $request, $modifier) {
    $uri = $request->getRequestUri();
    
    // Handle both /modifier and /modifier/ for exact matches
    if ($uri === '/' . $modifier || $uri === '/' . $modifier . '/') {
      return TRUE;
    }
    
    $result = $this->checkPath($modifier, $uri);
    return $result;
  }

  /**
   * @param $modifier
   * @param $uri
   *
   * @return bool
   */
  protected function checkPath($modifier, $uri) {
    // Don't process if this is the exact group path (with or without trailing slash)
    if ($uri === '/' . $modifier || $uri === '/' . $modifier . '/') {
      return FALSE;
    }
    return strpos($uri, '/' . $modifier . '/') === 0;
  }

  /**
   *
   */
  public function alterRequest(Request $request, $identifier) {
    // Cannot use $request->uri as this sets it to the current server URI, making
    // it too late to modify.
    $uri = $request->server->get('REQUEST_URI');
    // Exact match without trailing slash - let main request continue
    if ($uri === '/' . $identifier) {
      return FALSE;
    }
    
    // Exact match with trailing slash - redirect to remove trailing slash
    if ($uri === '/' . $identifier . '/') {
      return new TrustedRedirectResponse('/' . $identifier);
    }
    // If we try to get the base path from the Request argument, the modifier gets matched twice.
    // getBasePath() indirectly populates the requestUri parameter, which needs to be null before we set the
    // REQUEST_URI parameter.
    $pos = strpos($uri, '/' . $identifier);
    if ($pos !== FALSE) {
      $newPath = substr_replace($uri, '', $pos, strlen($identifier) + 1);

      if (strpos($newPath, '/' . $identifier . '/') === 0) {
        // ... then we have this path multiple times. Redirect up one.
        return new TrustedRedirectResponse($newPath);
      }
      if ($newPath === '/') {
        // ... then redirect to the group page.
        return new TrustedRedirectResponse('/' . $identifier);
      }
      $request->server->set('REQUEST_URI', $newPath);
      return $request;
    }

    return FALSE;
  }

  /**
   *
   */
  public function enterContext($modifier, $path, array &$options) {
    if (isset($options['purl_exit']) && $options['purl_exit']) {
      return $path;
    }
    
    // For canonical group routes, path is '/' - return just the modifier to avoid trailing slash
    if ($path === '/') {
      return '/' . $modifier;
    }
    
    return '/' . $modifier . $path;
  }

  /**
   *
   */
  public function exitContext($modifier, $path, array &$options) {
    if (!$this->checkPath($modifier, $path)) {
      return NULL;
    }

    return substr($path, 0, strlen($modifier) + 1);
  }

}
