<?php

declare(strict_types=1);

namespace Drupal\Tests\group_purl\Unit;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\group_purl\Plugin\Purl\Provider\GroupPurlProvider;
use Drupal\path_alias\AliasManagerInterface;

/**
 * Unit tests for GroupPurlProvider.
 *
 * @coversDefaultClass \Drupal\group_purl\Plugin\Purl\Provider\GroupPurlProvider
 * @group group_purl
 */
class GroupPurlProviderTest extends UnitTestCase {

  /**
   * The group PURL provider under test.
   *
   * @var \Drupal\group_purl\Plugin\Purl\Provider\GroupPurlProvider
   */
  protected GroupPurlProvider $provider;

  /**
   * Mock cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $cache;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * Mock alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $aliasManager;

  /**
   * Mock database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $connection;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->cache = $this->createMock(CacheBackendInterface::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->aliasManager = $this->createMock(AliasManagerInterface::class);
    $this->connection = $this->createMock(Connection::class);

    $this->provider = new GroupPurlProvider(
      $this->cache,
      $this->entityTypeManager,
      $this->aliasManager,
      $this->connection
    );
  }

  /**
   * Tests getModifierData with cached data.
   *
   * @covers ::getModifierData
   */
  public function testGetModifierDataWithCache(): void {
    $cached_data = [
      'business' => '2',
      'council' => '20',
      'education' => '5',
    ];

    $cache_item = (object) ['data' => $cached_data];
    $this->cache->expects($this->once())
      ->method('get')
      ->with(GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY)
      ->willReturn($cache_item);

    $result = $this->provider->getModifierData();
    $this->assertEquals($cached_data, $result);
  }

  /**
   * Tests getModifierData without cache.
   *
   * @covers ::getModifierData
   */
  public function testGetModifierDataWithoutCache(): void {
    $this->cache->expects($this->once())
      ->method('get')
      ->with(GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY)
      ->willReturn(FALSE);

    // Mock database query result.
    $query_result = $this->createMock('\Drupal\Core\Database\StatementInterface');
    $aliases = [
      (object) ['path' => '/group/2', 'alias' => '/business'],
      (object) ['path' => '/group/20', 'alias' => '/council'],
      (object) ['path' => '/group/5', 'alias' => '/education'],
    ];
    $query_result->expects($this->once())
      ->method('fetchAllAssoc')
      ->with('path')
      ->willReturn($aliases);

    $select = $this->createMock('\Drupal\Core\Database\Query\SelectInterface');
    $select->expects($this->once())
      ->method('fields')
      ->willReturnSelf();
    $select->expects($this->exactly(2))
      ->method('condition')
      ->willReturnSelf();
    $select->expects($this->once())
      ->method('distinct')
      ->willReturnSelf();
    $select->expects($this->once())
      ->method('execute')
      ->willReturn($query_result);

    $this->connection->expects($this->once())
      ->method('select')
      ->with('path_alias', 'pa')
      ->willReturn($select);

    $expected_data = [
      'business' => '2',
      'council' => '20',
      'education' => '5',
    ];

    $this->cache->expects($this->once())
      ->method('set')
      ->with(
        GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY,
        $expected_data,
        $this->anything(),
        [GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY]
      );

    $result = $this->provider->getModifierData();
    $this->assertEquals($expected_data, $result);
  }

  /**
   * Tests getModifierDataByKey.
   *
   * @covers ::getModifierDataByKey
   */
  public function testGetModifierDataByKey(): void {
    $cached_data = [
      'business' => '2',
      'council' => '20',
      'education' => '5',
    ];

    $cache_item = (object) ['data' => $cached_data];
    $this->cache->expects($this->once())
      ->method('get')
      ->with(GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY)
      ->willReturn($cache_item);

    $result = $this->provider->getModifierDataByKey('council');
    $this->assertEquals(['council' => '20'], $result);
  }

  /**
   * Tests getModifierDataByKey with non-existent key.
   *
   * @covers ::getModifierDataByKey
   */
  public function testGetModifierDataByKeyNotFound(): void {
    $this->cache->expects($this->once())
      ->method('get')
      ->willReturn(FALSE);

    $this->aliasManager->expects($this->once())
      ->method('getPathByAlias')
      ->with('/nonexistent')
      ->willReturn('/nonexistent');

    $result = $this->provider->getModifierDataByKey('nonexistent');
    $this->assertEquals([], $result);
  }

  /**
   * Tests getModifierDataById.
   *
   * @covers ::getModifierDataById
   */
  public function testGetModifierDataById(): void {
    $cached_data = [
      'business' => '2',
      'council' => '20',
      'education' => '5',
    ];

    $cache_item = (object) ['data' => $cached_data];
    $this->cache->expects($this->once())
      ->method('get')
      ->with(GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY)
      ->willReturn($cache_item);

    $result = $this->provider->getModifierDataById('20');
    $this->assertEquals(['council' => '20'], $result);
  }

  /**
   * Tests getModifierDataById with non-existent ID.
   *
   * @covers ::getModifierDataById
   */
  public function testGetModifierDataByIdNotFound(): void {
    $cached_data = [
      'business' => '2',
      'council' => '20',
    ];

    $cache_item = (object) ['data' => $cached_data];
    $this->cache->expects($this->once())
      ->method('get')
      ->with(GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY)
      ->willReturn($cache_item);

    // Mock the entity storage to return null for non-existent group.
    $storage = $this->createMock('\Drupal\Core\Entity\EntityStorageInterface');
    $storage->expects($this->once())
      ->method('load')
      ->with('999')
      ->willReturn(NULL);

    $this->entityTypeManager->expects($this->once())
      ->method('getStorage')
      ->with('group')
      ->willReturn($storage);

    $result = $this->provider->getModifierDataById('999');
    $this->assertEquals([], $result);
  }

  /**
   * Tests cache invalidation.
   *
   * @covers ::invalidateCache
   */
  public function testInvalidateCache(): void {
    $this->cache->expects($this->once())
      ->method('invalidate')
      ->with(GroupPurlProvider::GROUP_PURL_MODIFIERS_CACHE_KEY);

    $this->provider->invalidateCache();
  }

}
