<?php

declare(strict_types=1);

namespace Drupal\Tests\group_purl\Unit;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\group_purl\PathProcessor\GroupPurlUrlGenerationProcessor;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\purl\ContextHelper;
use Drupal\purl\MatchedModifiers;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for GroupPurlUrlGenerationProcessor.
 *
 * @coversDefaultClass \Drupal\group_purl\PathProcessor\GroupPurlUrlGenerationProcessor
 * @group group_purl
 */
class GroupPurlUrlGenerationProcessorTest extends UnitTestCase {

  /**
   * The processor under test.
   *
   * @var \Drupal\group_purl\PathProcessor\GroupPurlUrlGenerationProcessor
   */
  protected GroupPurlUrlGenerationProcessor $processor;

  /**
   * Mock matched modifiers.
   *
   * @var \Drupal\purl\MatchedModifiers|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $matchedModifiers;

  /**
   * Mock context helper.
   *
   * @var \Drupal\purl\ContextHelper|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $contextHelper;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * Mock alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $aliasManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->matchedModifiers = $this->createMock(MatchedModifiers::class);
    $this->contextHelper = $this->createMock(ContextHelper::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->aliasManager = $this->createMock(AliasManagerInterface::class);

    $this->processor = new GroupPurlUrlGenerationProcessor(
      $this->matchedModifiers,
      $this->contextHelper,
      $this->entityTypeManager,
      $this->aliasManager
    );
  }

  /**
   * Tests that system routes are excluded from processing entirely.
   *
   * This test verifies the fix for the issue where group_purl was interfering
   * with private file system routes and other system routes.
   *
   * @covers ::processOutbound
   * @dataProvider systemRoutesProvider
   */
  public function testSystemRoutesSkippedCompletely(string $path): void {
    // No methods on dependencies should be called for system routes.
    $this->entityTypeManager->expects($this->never())->method($this->anything());
    $this->aliasManager->expects($this->never())->method($this->anything());
    $this->matchedModifiers->expects($this->never())->method($this->anything());
    $this->contextHelper->expects($this->never())->method($this->anything());

    $options = [];
    $result = $this->processor->processOutbound($path, $options);

    // The path should be returned unchanged.
    $this->assertEquals($path, $result);

    // No options should be modified.
    $this->assertEmpty($options);
  }

  /**
   * Provides test cases for various system routes.
   */
  public static function systemRoutesProvider(): array {
    return [
      'Private file download' => ['/system/files/private/document.pdf'],
      'Public file download (non-standard)' => ['/system/files/public/image.jpg'],
      'System file with ID pattern' => ['/system/files/123'],
      'System AJAX route' => ['/system/ajax'],
      'System batch route' => ['/system/batch'],
      'System cron route' => ['/system/cron/abc123'],
      'System 404 route' => ['/system/404'],
      'System 403 route' => ['/system/403'],
      'System 401 route' => ['/system/401'],
      'Complex system file path' => ['/system/files/private/uploads/2024/document.pdf'],
      'System temporary files' => ['/system/files/temporary/temp.txt'],
    ];
  }

  /**
   * Tests the fix prevents the original error with private file routes.
   *
   * This specifically tests that paths matching the pattern /system/files/123
   * don't cause "The 'system' entity type does not exist" errors.
   *
   * @covers ::processOutbound
   */
  public function testPrivateFileRouteErrorPrevention(): void {
    // This path would have previously caused the error because it matches
    // the regex pattern /(\w+)/(\d+)/ with "system" and "123".
    $problematic_path = '/system/files/123';

    // No methods should be called because we exit early for system routes.
    $this->entityTypeManager->expects($this->never())->method($this->anything());
    $this->aliasManager->expects($this->never())->method($this->anything());

    $options = [];
    $result = $this->processor->processOutbound($problematic_path, $options);

    // Should return the path unchanged without any processing.
    $this->assertEquals($problematic_path, $result);
    $this->assertEmpty($options);
  }

}
