# Group PURL

Integrates the Group module with PURL (Persistent URLs) to provide group-specific URL prefixes and context-aware routing for Drupal.

## Requirements

- Drupal 10.3+ or Drupal 11+
- Group 3.x module
- PURL module
- Path Alias module (core)

## Overview

Group PURL enables automatic URL prefixing for group content, ensuring that content belonging to groups appears under the appropriate group context. For example, a node `/business-meetings` belonging to a group with alias `/business` will automatically be accessible at `/business/business-meetings`.

## Routing Requirements

The module implements 5 key routing behaviors:

1. **Correct group context proceeds normally**: When accessing group content with the correct group prefix, content renders normally
2. **Missing group prefix redirects to group context**: Direct access to group content (e.g., `/content-page`) redirects to the group context (e.g., `/group-name/content-page`)
3. **Wrong group prefix redirects to correct group**: Accessing group content with an incorrect group prefix redirects to the correct group context
4. **Entity links get group prefixes automatically**: Links generated via `$entity->toUrl()` automatically include group prefixes when appropriate
5. **Menu links respect PURL context settings**: Menu links honor their PURL fieldset configuration for context handling

## Configuration

### 1. Group Setup
1. Create groups and assign them path aliases (e.g., `/group/20` → `/council`)
2. Add content to groups using Group's content management interface

### 2. Content Type Configuration
For content types that should maintain group context:

1. Navigate to **Structure → Content types → [Content Type] → Edit**
2. Expand **PURL settings** fieldset
3. Check **"Keep context of the node"**
4. Save the content type

This setting determines whether content of this type should:
- **Checked**: Stay in group context (redirect to group URL if accessed directly)
- **Unchecked**: Be accessible both inside and outside group context

### 3. Menu Link Configuration
When editing menu links, use the **Persistent URLs** fieldset to control context behavior:

- **Maintain context**: Link inherits current PURL context (default)
- **Strip PURL context**: Link always goes to non-group version
- **Specific group context**: Link always goes to specified group context

### 4. PURL Provider Configuration
Ensure your group PURL provider is configured with:
- **Method**: Group Content
- **Modifiers**: Generated automatically from group aliases

## Technical Implementation

### Path Processing
- **Inbound**: PURL processes group prefixes and creates subrequests
- **Outbound**: Group PURL automatically adds prefixes to entity URLs when appropriate

### Service Priority
- PURL outbound processor (priority 300) runs before Group PURL (priority 10)
- This ensures explicit menu contexts are handled before automatic group detection

### Event Subscribers
- **NodeGroupViewController**: Handles redirect logic for nodes with keep_context setting
- **GroupContextRouteSubscriber**: Manages group context detection and validation

## Debugging

Enable debug logging by adding to settings.php:
```php
$config['system.logging']['error_level'] = 'verbose';
```

Then monitor logs for 'group_purl' entries:
```bash
drush watchdog:show --filter=group_purl
```

## Testing

Run the included tests:
```bash
# Unit tests
./vendor/bin/phpunit web/modules/contrib/group_purl/tests/src/Unit/

# Functional tests
./vendor/bin/phpunit web/modules/contrib/group_purl/tests/src/Functional/
```

## Troubleshooting

### Content not redirecting to group context
1. Verify content type has "Keep context of the node" enabled
2. Confirm content is actually added to a group
3. Check that group has a proper path alias set

### Menu links not respecting PURL settings
1. Verify PURL fieldset appears on menu link edit forms
2. If missing with menu_item_extras, ensure form alter hooks are working
3. Clear cache after configuration changes

### Double prefixes in URLs
1. Check service priority configuration in group_purl.services.yml
2. Verify path processor logic isn't conflicting with PURL's processing

## API

### Custom URL Generation
```php
// Get entity URL with automatic group prefix
$url = $entity->toUrl()->toString();

// Force specific group context
$url = $entity->toUrl()->setOption('purl_context', [
  'provider' => 'group_purl_provider',
  'modifier' => 'group-alias'
])->toString();

// Strip PURL context
$url = $entity->toUrl()->setOption('purl_context', FALSE)->toString();
```

### Group Context Detection
```php
$group_context = \Drupal::service('group_purl.context_provider');
$current_group = $group_context->getCurrentGroup();
```

## Compatibility

- **Drupal**: 10.3+ and 11+
- **PHP**: 8.1+
- **Group**: 3.x
- **PURL**: Latest stable version

## Group Prefix/Context Flow

The following diagram illustrates how Group PURL processes URLs and manages group context:

```mermaid
flowchart TD
    A[URL Request] --> B{Group PURL Path Processor<br/>Priority 310}
    B --> C{Path Type?}

    C -->|Group Entity Route<br/>/group/123| D{Group Route Analysis}
    C -->|Entity Route<br/>/node/456| E{Entity Analysis}
    C -->|Other Route| F[Pass Through]

    D --> G{Current vs Target Group?}
    G -->|Same Group + Canonical| H[Let PURL Handle Normally]
    G -->|Same Group + Non-canonical| I[Set Context & Strip Prefix]
    G -->|Different Group + Canonical| J[Exit PURL Context<br/>purl_context = FALSE]
    G -->|Different Group + Non-canonical| K[Switch Context<br/>provider + modifier]
    G -->|No Current Context + Canonical| L[Let Normal Routing Handle]
    G -->|No Current Context + Non-canonical| M[Enter Context<br/>provider + modifier]

    E --> N{Entity in Group?}
    N -->|No| O[No Prefix Applied]
    N -->|Yes| P{Keep Context Setting?}
    P -->|No| O
    P -->|Yes| Q{Current vs Entity Group?}
    Q -->|Same Group| R[Maintain Context<br/>provider + modifier]
    Q -->|Different Group| S[Switch Context<br/>provider + modifier]
    Q -->|No Current Context| T[Enter Context<br/>provider + modifier]

    H --> U[PURL Processor<br/>Priority 300]
    I --> V[Strip + Set Context]
    J --> W[Strip Prefix Only]
    K --> V
    L --> U
    M --> V
    O --> U
    R --> V
    S --> V
    T --> V
    F --> U

    V --> U
    W --> U
    U --> X[Final URL Generated]

    style B fill:#e1f5fe
    style U fill:#f3e5f5
    style X fill:#e8f5e8
```

### Flow Explanation

1. **Group PURL Path Processor (Priority 310)** runs first to:
   - Analyze the path type (group entity, content entity, or other)
   - Check current group context vs target entity's group
   - Set appropriate `purl_context` options for PURL

2. **Context Decision Logic**:
   - **Same Group Canonical**: Let PURL handle normally (avoids double prefixes)
   - **Different Groups**: Either exit context entirely or switch context
   - **Entity Links**: Check if entity is in group and should keep context
   - **Menu Links**: Override "preserve context" with correct entity context

3. **PURL Processor (Priority 300)** runs second to:
   - Process the `purl_context` options set by Group PURL
   - Add/remove URL prefixes as appropriate
   - Generate final URLs

This ensures menu links always navigate to correct group contexts while avoiding double prefixes and respecting content type settings.

## Development

### Running GitLab CI Tests
The module includes GitLab CI configuration for automated testing across multiple Drupal and PHP versions. Tests run automatically on pushes and merge requests.

### Code Standards
Code follows Drupal coding standards and includes:
- Proper type hints and return types
- Service dependency injection
- Event-driven architecture
- Comprehensive logging for debugging
