<?php

namespace Drupal\group_purl\Context;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\ContextProviderInterface;
use Drupal\Core\Plugin\Context\EntityContext;
use Drupal\Core\Routing\CurrentRouteMatch;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\purl\Event\ModifierMatchedEvent;
use Drupal\purl\PurlEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Class GroupPurlContext.
 */
class GroupPurlContext implements ContextProviderInterface, EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Drupal\Core\Entity\EntityTypeManager definition.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current route match service.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $currentRouteMatch;

  /**
   * The PURL matched modifiers service.
   *
   * @var \Drupal\purl\MatchedModifiers
   */
  protected $matchedModifiers;

  /**
   * The modifier matched event.
   *
   * @var \Drupal\purl\Event\ModifierMatchedEvent|null
   */
  protected $modifierMatched;

  /**
   * Contexts collected from events.
   *
   * @var array
   */
  protected $contexts = [];

  /**
   * Constructs a new GroupPurlContext object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Routing\CurrentRouteMatch $currentRouteMatch
   *   The current route match service.
   * @param \Drupal\purl\MatchedModifiers $matchedModifiers
   *   The PURL matched modifiers service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, CurrentRouteMatch $currentRouteMatch, $matchedModifiers = NULL) {
    $this->entityTypeManager = $entity_type_manager;
    $this->currentRouteMatch = $currentRouteMatch;
    $this->matchedModifiers = $matchedModifiers;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[PurlEvents::MODIFIER_MATCHED] = ['onModifierMatched'];
    return $events;
  }

  /**
   * This method is called whenever the purl.modifier_matched event is
   * dispatched.
   *
   * @param \Drupal\purl\Event\ModifierMatchedEvent $event
   *   The event object.
   */
  public function onModifierMatched(ModifierMatchedEvent $event) {
    \Drupal::logger('group_purl')->debug('GroupPurlContext::onModifierMatched() - Method: @method, Modifier: @modifier, Value: @value', [
      '@method' => $event->getMethod()->getId(),
      '@modifier' => $event->getModifier(),
      '@value' => $event->getValue(),
    ]);
    $this->modifierMatched = $event;
    $this->contexts[$event->getMethod()->getId()] = $event->getModifier();
    ksort($this->contexts);
  }

  /**
   * {@inheritdoc}
   */
  public function getRuntimeContexts(array $unqualified_context_ids) {
    $cacheability = new CacheableMetadata();
    $cacheability->setCacheContexts(['purl']);

    $group = $this->getGroupFromRoute();
    if ($group) {
      $context = EntityContext::fromEntity($group);
      $context->addCacheableDependency($cacheability);
      return ['group' => $context];
    }
    return [];
  }

  /**
   * Retrieves the active group from the route or modifier.
   *
   * @return \Drupal\group\Entity\Group|null
   *   The active group entity, or NULL if none is found.
   */
  public function getGroupFromRoute() {
    // Try to get matched modifiers from request attributes (for subrequests)
    $request = \Drupal::request();
    $request_modifiers = $request->attributes->get('purl.matched_modifiers', []);

    \Drupal::logger('group_purl')->debug('GroupPurlContext::getGroupFromRoute() - Request modifiers: @count', ['@count' => count($request_modifiers)]);

    foreach ($request_modifiers as $match) {
      if (isset($match['method']) && $match['method']->getPluginId() === 'group_prefix') {
        $storage = $this->entityTypeManager->getStorage('group');
        $group_id = $match['value'];
        \Drupal::logger('group_purl')->debug('GroupPurlContext: Found group_prefix in request attributes with group ID: @id', ['@id' => $group_id]);
        $group = $storage->load($group_id);
        if ($group) {
          \Drupal::logger('group_purl')->debug('GroupPurlContext: Loaded group from request: @label (@id)', ['@label' => $group->label(), '@id' => $group->id()]);
          return $group;
        }
      }
    }

    // Fallback: try to get matched modifiers from PURL's service.
    $matched_modifiers_service = $this->matchedModifiers ?: \Drupal::service('purl.matched_modifiers');
    $matched = $matched_modifiers_service->getMatched();

    \Drupal::logger('group_purl')->debug('GroupPurlContext::getGroupFromRoute() - PURL matched modifiers service: @count', ['@count' => count($matched)]);

    foreach ($matched as $modifier) {
      if ($modifier->getMethod()->getPluginId() === 'group_prefix') {
        $storage = $this->entityTypeManager->getStorage('group');
        $group_id = $modifier->getValue();
        \Drupal::logger('group_purl')->debug('GroupPurlContext: Found group_prefix modifier with group ID: @id', ['@id' => $group_id]);
        $group = $storage->load($group_id);
        if ($group) {
          \Drupal::logger('group_purl')->debug('GroupPurlContext: Loaded group: @label (@id)', ['@label' => $group->label(), '@id' => $group->id()]);
          return $group;
        }
      }
    }

    // Fallback to our event-based approach (for main requests)
    \Drupal::logger('group_purl')->debug('GroupPurlContext::getGroupFromRoute() - modifierMatched is @null', ['@null' => $this->modifierMatched ? 'NOT NULL' : 'NULL']);
    if ($this->modifierMatched !== NULL) {
      $storage = $this->entityTypeManager->getStorage('group');
      $modifier_value = $this->modifierMatched->getValue();
      \Drupal::logger('group_purl')->debug('GroupPurlContext: Modifier matched, value: @value', ['@value' => $modifier_value]);
      $group = $storage->load($modifier_value);
      \Drupal::logger('group_purl')->debug('GroupPurlContext: Loaded group: @group', ['@group' => $group ? $group->id() : 'none']);
      return $group;
    }

    $routename = $this->currentRouteMatch->getRouteName();
    if ($routename !== NULL && strpos($routename, 'entity.group.') === 0) {
      $group = $this->currentRouteMatch->getParameter('group');
      return $group;
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableContexts() {
    $context = EntityContext::fromEntityTypeId('group', $this->t('Group from Purl'));
    return ['group' => $context];
  }

}
