<?php

namespace Drupal\group_purl\EventSubscriber;

use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Response;
use Drupal\group\Entity\GroupRelationship;
use Drupal\purl\Event\ExitedContextEvent;
use Drupal\purl\PurlEvents;
use Drupal\redirect\Exception\RedirectLoopException;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\CurrentRouteMatch;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\purl\MatchedModifiers;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Drupal\Core\Routing\NullRouteMatch;

/**
 * Class GroupContextRouteSubscriber.
 */
class GroupContextRouteSubscriber implements EventSubscriberInterface {

  /**
   * Drupal\Core\Entity\EntityTypeManagerInterface definition.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;
  /**
   * Drupal\Core\Routing\CurrentRouteMatch definition.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $currentRouteMatch;
  /**
   * Drupal\purl\MatchedModifiers definition.
   *
   * @var \Drupal\purl\MatchedModifiers
   */
  protected $purlMatchedModifiers;

  /**
   * The path alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;

  /**
   * Constructs a new GroupContextRouteSubscriber object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, CurrentRouteMatch $current_route_match, MatchedModifiers $purl_matched_modifiers, AliasManagerInterface $alias_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->currentRouteMatch = $current_route_match;
    $this->purlMatchedModifiers = $purl_matched_modifiers;
    $this->aliasManager = $alias_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Temporarily disable while debugging redirect issues
    // $events[KernelEvents::REQUEST][] = ['checkGroupContext', -50];.
    return [];
  }

  /**
   * This method is called whenever the KernelEvents::REQUEST event is
   * dispatched.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   * @param $eventName
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   */
  public function checkGroupContext(RequestEvent $event, $eventName, EventDispatcherInterface $eventDispatcher) {

    $route_match = $this->currentRouteMatch;
    $master_route_match = $this->currentRouteMatch->getMasterRouteMatch();
    if (!$master_route_match instanceof NullRouteMatch) {
      $route_match = $master_route_match;
    }

    $route_options = $route_match->getRouteObject()->getOptions();
    $isAdminRoute = array_key_exists('_admin_route', $route_options);
    $route_name = $route_match->getRouteName();

    $matched = $this->purlMatchedModifiers->getMatched();
    $url = FALSE;
    $multiple = count($matched) > 1;

    if (empty($matched)) {
      if ($this->shouldMaintainGroupContext($route_name, $route_match)) {
        $entity = $this->getEntityFromRoute($route_match);
        if ($entity && ($contents = GroupRelationship::loadByEntity($entity))) {
          $group_content = reset($contents);
          $group = $group_content->getGroup();
          $alias = $this->aliasManager->getAliasByPath('/group/' . $group->id());
          $modifier = substr($alias, 1);

          // Only redirect if we have a valid modifier and it's different from current context.
          if ($modifier && $modifier !== '/group/' . $group->id()) {
            $url = $this->buildGroupContextUrl($route_name, $route_match, $modifier, $entity);
          }
        }
      }
      elseif ($route_name == 'entity.group.canonical') {

      }
      elseif ($isAdminRoute) {
        return;
      }
    }
    else {
      if ($this->shouldMaintainGroupContext($route_name, $route_match)) {
        $entity = $this->getEntityFromRoute($route_match);
        if ($entity && ($contents = GroupRelationship::loadByEntity($entity))) {
          $group_content = reset($contents);
          $group = $group_content->getGroup();
          $alias = $this->aliasManager->getAliasByPath('/group/' . $group->id());
          $modifier = substr($alias, 1);

          // Check if current context matches the expected group.
          $current_modifier = reset($matched);
          $expected_group_id = $group->id();

          if ($multiple || !$this->contextMatches($current_modifier, $expected_group_id)) {
            $url = $this->buildGroupContextUrl($route_name, $route_match, $modifier, $entity);
          }
          // Else no redirect needed, success.
        }
        else {
          // This node is not in a group.
          $url = Url::fromRoute($route_name, $route_match->getRawParameters()->all(), [
            'purl_exit' => TRUE,
          ]);
        }
      }
      elseif ($route_name == 'entity.group.canonical') {
        /** @var \Drupal\group\Entity\Group $group */
        $group = $route_match->getParameter('group');
        $alias = $this->aliasManager->getAliasByPath('/group/' . $group->id());
        $modifier = substr($alias, 1);

      }
      elseif ($isAdminRoute) {
        // Exit group
        // $url = Url::fromRoute($route_name, $this->currentRouteMatch->getRawParameters()->all(), [
        //   'purl_exit' => TRUE,
        //  ]);.
      }

    }
    // If ($route_name == 'entity.group.canonical') {
    //  if (empty($matched)) {
    //    $matched = 1;
    //  }
    //  if (empty($matched)) {.
    // $url = Url::fromRoute($route_name, $this->currentRouteMatch->getRawParameters()
    //      ->all(), [
    //      'host' => $modifier . '.' . Settings::get('purl_base_domain'),
    //      'absolute' => TRUE,
    //      'purl_exit' => TRUE,
    //    ]);
    //  }
    // }.
    if ($url && $event->getRequest()->getUri() != $url->toString()) {
      try {
        $url_string = $url->toString();
        // Prevent malformed URLs by ensuring we have a proper domain.
        if (!preg_match('/^https?:\/\//', $url_string)) {
          // If URL doesn't start with protocol, make it absolute with current host.
          $request = $event->getRequest();
          $base_url = $request->getScheme() . '://' . $request->getHttpHost();
          if (!str_starts_with($url_string, '/')) {
            $url_string = '/' . $url_string;
          }
          $url_string = $base_url . $url_string;
        }

        $redirect_response = new TrustedRedirectResponse($url_string);
        $redirect_response->getCacheableMetadata()->setCacheMaxAge(0);
        $modifiers = $event->getRequest()->attributes->get('purl.matched_modifiers', []);
        $new_event = new ExitedContextEvent($event->getRequest(), $redirect_response, $this->currentRouteMatch, $modifiers);
        $eventDispatcher->dispatch($new_event, PurlEvents::EXITED_CONTEXT);
        $event->setResponse($new_event->getResponse());
        return;
      }
      catch (RedirectLoopException $e) {
        \Drupal::logger('group_purl')->warning('Redirect loop prevented: @message', ['@message' => $e->getMessage()]);
        $response = new Response();
        $response->setStatusCode(503);
        $response->setContent('Service unavailable');
        $event->setResponse($response);
        return;
      }
    }
  }

  /**
   * Check if the current PURL context matches the expected group.
   *
   * @param mixed $current_modifier
   *   The current PURL modifier.
   * @param string $expected_group_id
   *   The expected group ID.
   *
   * @return bool
   *   TRUE if contexts match, FALSE otherwise.
   */
  private function contextMatches($current_modifier, $expected_group_id): bool {
    if (!$current_modifier || !$expected_group_id) {
      return FALSE;
    }

    // Get the group ID from the current modifier.
    $provider = \Drupal::service('purl.plugin.provider_manager')
      ->createInstance('group_purl_provider');

    $modifier_data = $provider->getModifierDataByKey($current_modifier->getModifier());

    if (empty($modifier_data)) {
      return FALSE;
    }

    $current_group_id = reset($modifier_data);
    return $current_group_id == $expected_group_id;
  }

  /**
   * Check if a route should maintain group context.
   *
   * @param string $route_name
   *   The route name.
   * @param \Drupal\Core\Routing\CurrentRouteMatch $route_match
   *   The route match.
   *
   * @return bool
   *   TRUE if the route should maintain group context.
   */
  private function shouldMaintainGroupContext(string $route_name, $route_match): bool {
    // Define routes that should maintain group context.
    $group_context_routes = [
      'entity.node.canonical',
      'entity.node.edit_form',
      'entity.node.delete_form',
      'entity.node.add',
      'entity.media.canonical',
      'entity.media.edit_form',
      'entity.media.delete_form',
      'entity.user.canonical',
      'entity.user.edit_form',
      // Add other entity routes as needed.
    ];

    return in_array($route_name, $group_context_routes);
  }

  /**
   * Get the entity from the route match.
   *
   * @param \Drupal\Core\Routing\CurrentRouteMatch $route_match
   *   The route match.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The entity if found, NULL otherwise.
   */
  private function getEntityFromRoute($route_match) {
    // Try different entity parameter names.
    $entity_parameters = ['node', 'media', 'user', 'taxonomy_term', 'entity'];

    foreach ($entity_parameters as $param) {
      if ($entity = $route_match->getParameter($param)) {
        return $entity;
      }
    }

    return NULL;
  }

  /**
   * Build a group context URL for redirecting.
   *
   * @param string $route_name
   *   The route name.
   * @param \Drupal\Core\Routing\CurrentRouteMatch $route_match
   *   The route match.
   * @param string $modifier
   *   The PURL modifier (group alias without leading slash).
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return \Drupal\Core\Url|null
   *   The URL to redirect to, or NULL if none needed.
   */
  private function buildGroupContextUrl(string $route_name, $route_match, string $modifier, $entity) {
    $request = \Drupal::request();

    // Handle different route types.
    switch ($route_name) {
      case 'entity.node.canonical':
      case 'entity.media.canonical':
      case 'entity.user.canonical':
        // For canonical routes, use the entity's alias.
        $entity_path = '/' . $entity->getEntityTypeId() . '/' . $entity->id();
        $entity_alias = $this->aliasManager->getAliasByPath($entity_path);
        $prefixed_path = '/' . $modifier . $entity_alias;
        break;

      case 'entity.node.edit_form':
      case 'entity.media.edit_form':
      case 'entity.user.edit_form':
        // For edit forms, use the edit path.
        $prefixed_path = '/' . $modifier . '/' . $entity->getEntityTypeId() . '/' . $entity->id() . '/edit';
        break;

      case 'entity.node.delete_form':
      case 'entity.media.delete_form':
        // For delete forms, use the delete path.
        $prefixed_path = '/' . $modifier . '/' . $entity->getEntityTypeId() . '/' . $entity->id() . '/delete';
        break;

      default:
        // For other routes, use the current path.
        $prefixed_path = '/' . $modifier . $request->getPathInfo();
        break;
    }

    return Url::fromUserInput($prefixed_path, [
      'query' => $request->query->all(),
    ]);
  }

}
