<?php

namespace Drupal\group_purl\PathProcessor;

use Drupal\Core\PathProcessor\OutboundPathProcessorInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\purl\MatchedModifiers;
use Drupal\purl\ContextHelper;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\group\Entity\GroupRelationship;
use Drupal\path_alias\AliasManagerInterface;

/**
 * Class GroupPurlUrlGenerationProcessor.
 *
 * Handles outbound URL generation for group entities and group content.
 * Runs BEFORE PURL processor to set purl_context options.
 */
class GroupPurlUrlGenerationProcessor implements OutboundPathProcessorInterface {

  /**
   * Drupal\purl\MatchedModifiers definition.
   *
   * @var \Drupal\purl\MatchedModifiers
   */
  protected $purlMatchedModifiers;

  /**
   * Drupal\purl\ContextHelper definition.
   *
   * @var \Drupal\purl\ContextHelper
   */
  protected $purlContextHelper;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;

  /**
   * Constructs a new GroupPurlUrlGenerationProcessor object.
   */
  public function __construct(MatchedModifiers $purl_matched_modifiers, ContextHelper $purl_context_helper, EntityTypeManagerInterface $entity_type_manager, AliasManagerInterface $alias_manager) {
    $this->purlMatchedModifiers = $purl_matched_modifiers;
    $this->purlContextHelper = $purl_context_helper;
    $this->entityTypeManager = $entity_type_manager;
    $this->aliasManager = $alias_manager;
  }

  /**
   * @param string $path
   * @param array $options
   * @param \Symfony\Component\HttpFoundation\Request|null $request
   * @param \Drupal\Core\Render\BubbleableMetadata|null $bubbleable_metadata
   *
   * @return string
   *
   *   This processor sets PURL context for entity URL generation.
   *   It runs BEFORE PURL processor to set purl_context options.
   */
  public function processOutbound($path, &$options = [], ?Request $request = NULL, ?BubbleableMetadata $bubbleable_metadata = NULL) {
    // EARLY EXIT: Skip all system routes to prevent interference with file downloads, AJAX, etc.
    if (str_starts_with($path, '/system/')) {
      return $path;
    }

    // FIRST: Check if this path has nested group contexts and clean them up.
    $cleaned_path = $this->removeNestedGroupContexts($path);
    if ($cleaned_path !== $path) {
      \Drupal::logger('group_purl')->error('GroupPurlUrlGenerationProcessor: Cleaned nested contexts from @original to @cleaned', [
        '@original' => $path,
        '@cleaned' => $cleaned_path,
      ]);
      $path = $cleaned_path;
    }

    // Only process entity URLs for URL generation.
    $entity_info = $this->getEntityInfoFromPath($path);

    if ($entity_info) {
      return $this->processEntityUrlGeneration($path, $options, $entity_info);
    }

    // For non-entity routes, apply menu link context rules.
    return $this->processNonEntityRoute($path, $options);
  }

  /**
   * Removes nested group contexts from a path.
   *
   * For example: /tourism/business/development-plan -> /development-plan
   * The correct group context will be determined by entity relationships.
   */
  protected function removeNestedGroupContexts($path) {
    // Get all group aliases to detect nested contexts.
    $group_aliases = $this->getAllGroupAliases();

    // Check if path starts with any group aliases.
    $path_parts = explode('/', trim($path, '/'));
    $cleaned_parts = [];
    $found_contexts = [];

    foreach ($path_parts as $part) {
      if (in_array($part, $group_aliases)) {
        $found_contexts[] = $part;
        // Skip group context parts, but remember them.
        continue;
      }
      else {
        // This is not a group context, keep the rest of the path.
        $cleaned_parts[] = $part;
      }
    }

    // If we found multiple group contexts, that's nested - clean it up.
    if (count($found_contexts) > 1) {
      \Drupal::logger('group_purl')->debug('GroupPurlUrlGenerationProcessor: Found nested group contexts: @contexts', [
        '@contexts' => implode(', ', $found_contexts),
      ]);
      return '/' . implode('/', $cleaned_parts);
    }

    // No nested contexts found, return original path.
    return $path;
  }

  /**
   * Gets all group aliases for nested context detection.
   */
  protected function getAllGroupAliases() {
    static $aliases = NULL;

    if ($aliases === NULL) {
      $aliases = [];

      try {
        $groups = $this->entityTypeManager->getStorage('group')->loadMultiple();
        foreach ($groups as $group) {
          $group_alias = $this->aliasManager->getAliasByPath('/group/' . $group->id());
          $group_modifier = ltrim($group_alias, '/');

          // Only include meaningful aliases (not system paths)
          if ($group_alias !== '/group/' . $group->id() && $group_modifier !== 'group/' . $group->id()) {
            $aliases[] = $group_modifier;
          }
        }
      }
      catch (\Exception $e) {
        \Drupal::logger('group_purl')->error('GroupPurlUrlGenerationProcessor: Error loading group aliases: @message', [
          '@message' => $e->getMessage(),
        ]);
      }
    }

    return $aliases;
  }

  /**
   * Gets entity information from a path.
   */
  protected function getEntityInfoFromPath($path) {
    // Check if path is a system entity path (e.g., /node/123, /node/123/edit)
    if (preg_match('#^/(\w+)/(\d+)(/.*)?$#', $path, $matches)) {
      $entity_type = $matches[1];

      // Validate that the entity type actually exists.
      if (!$this->entityTypeManager->hasDefinition($entity_type)) {
        return NULL;
      }

      return [
        'entity_type' => $entity_type,
        'entity_id' => $matches[2],
        'route_suffix' => $matches[3] ?? '',
      ];
    }

    // Check if path is an alias - convert to system path.
    $system_path = $this->aliasManager->getPathByAlias($path);
    if ($system_path !== $path && preg_match('#^/(\w+)/(\d+)(/.*)?$#', $system_path, $matches)) {
      $entity_type = $matches[1];

      // Validate that the entity type actually exists.
      if (!$this->entityTypeManager->hasDefinition($entity_type)) {
        return NULL;
      }

      return [
        'entity_type' => $entity_type,
        'entity_id' => $matches[2],
        'route_suffix' => $matches[3] ?? '',
      ];
    }

    return NULL;
  }

  /**
   * Processes entity URL generation by setting appropriate PURL context.
   */
  protected function processEntityUrlGeneration($path, &$options, $entity_info) {
    $entity_type = $entity_info['entity_type'];
    $entity_id = $entity_info['entity_id'];
    $route_suffix = $entity_info['route_suffix'];

    try {
      $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
      if (!$entity) {
        return $path;
      }

      // RULE: Group entities should use group context.
      if ($entity_type === 'group') {
        return $this->processGroupEntityUrl($path, $options, $entity, $route_suffix);
      }

      // RULE: User entities should never keep group context (even if they're group members)
      if ($entity_type === 'user') {
        $options['purl_context'] = FALSE;
        return $path;
      }

      // RULE: Check if entity is group content (in a group)
      $group_contents = GroupRelationship::loadByEntity($entity);
      if (!empty($group_contents)) {
        return $this->processGroupContentUrl($path, $options, $entity, $group_contents, $route_suffix);
      }

      // RULE: Non-group entities should remove group prefix (exit context)
      $options['purl_context'] = FALSE;
      return $path;

    }
    catch (\Exception $e) {
      \Drupal::logger('group_purl')->error('GroupPurlUrlGenerationProcessor: Error processing entity URL: @message', [
        '@message' => $e->getMessage(),
      ]);
      return $path;
    }
  }

  /**
   * Processes group entity URLs.
   */
  protected function processGroupEntityUrl($path, &$options, $group, $route_suffix) {
    $group_id = $group->id();
    $group_alias = $this->aliasManager->getAliasByPath('/group/' . $group_id);
    $group_modifier = ltrim($group_alias, '/');

    // If group has no meaningful alias, exit PURL context.
    if ($group_alias === '/group/' . $group_id || $group_modifier === 'group/' . $group_id) {
      $options['purl_context'] = FALSE;
      return $path;
    }

    // For canonical group path (/group/ID -> /business)
    if (empty($route_suffix) || $route_suffix === '/') {
      $options['purl_context'] = ['id' => $group_id];
      // Return root path so PURL processor creates /business, not /business/business.
      return '/';
    }
    else {
      // For non-canonical paths (/group/ID/edit -> /business/group/ID/edit)
      $options['purl_context'] = ['id' => $group_id];
      return '/group/' . $group_id . $route_suffix;
    }
  }

  /**
   * Processes group content URLs.
   */
  protected function processGroupContentUrl($path, &$options, $entity, $group_contents, $route_suffix) {
    // Use the first group (handle multiple group memberships by using first one)
    $group_content = reset($group_contents);
    $group = $group_content->getGroup();
    $group_id = $group->id();

    // Check if entity type should keep context (for nodes)
    if ($entity->getEntityTypeId() === 'node') {
      $node_type = $this->entityTypeManager->getStorage('node_type')->load($entity->bundle());
      $purl_settings = $node_type->getThirdPartySettings('purl');
      $keep_context = isset($purl_settings['keep_context']) && $purl_settings['keep_context'];

      if (!$keep_context) {
        $options['purl_context'] = FALSE;
        return $path;
      }
    }

    // RULE: Group content always uses its group's context.
    $group_alias = $this->aliasManager->getAliasByPath('/group/' . $group_id);
    $group_modifier = ltrim($group_alias, '/');

    // Don't add prefix if group has no meaningful alias.
    if ($group_alias === '/group/' . $group_id || $group_modifier === 'group/' . $group_id) {
      $options['purl_context'] = FALSE;
      return $path;
    }

    $options['purl_context'] = ['id' => $group_id];
    return $path;
  }

  /**
   * Processes non-entity routes using menu link context rules.
   */
  protected function processNonEntityRoute($path, &$options) {
    // RULE: Front page (home/logo links) should always exit group context.
    if ($path === '/' || $path === '<front>' || $this->isFrontPagePath($path)) {
      $options['purl_context'] = FALSE;
      return $path;
    }

    // Handle explicit purl_context from menu links.
    if (isset($options['purl_context'])) {
      // purl_context === false means "Exit PURL context" was selected.
      if ($options['purl_context'] === FALSE) {
        return $path;
      }

      // purl_context with id means specific context was selected.
      if (is_array($options['purl_context']) && isset($options['purl_context']['id'])) {
        return $path;
      }
    }

    // Default behavior: "Keep PURL context" - maintain current group context.
    $current_context = $this->getCurrentGroupContext();
    if ($current_context) {
      try {
        $current_group = $this->entityTypeManager->getStorage('group')->load($current_context);
        if ($current_group) {
          $group_alias = $this->aliasManager->getAliasByPath('/group/' . $current_group->id());
          $group_modifier = ltrim($group_alias, '/');

          if ($group_alias !== '/group/' . $current_group->id() && $group_modifier !== 'group/' . $current_group->id()) {
            $options['purl_context'] = ['id' => $current_group->id()];
            return $path;
          }
        }
      }
      catch (\Exception $e) {
        \Drupal::logger('group_purl')->error('GroupPurlUrlGenerationProcessor: Error maintaining context: @message', [
          '@message' => $e->getMessage(),
        ]);
      }
    }

    // No current context or couldn't maintain it.
    return $path;
  }

  /**
   * Gets the current group context from PURL matched modifiers.
   */
  protected function getCurrentGroupContext() {
    $matched = $this->purlMatchedModifiers->getMatched();

    foreach ($matched as $modifier) {
      if ($modifier->getMethod()->getPluginId() === 'group_prefix') {
        return $modifier->getValue();
      }
    }

    return NULL;
  }

  /**
   * Checks if a path is the configured front page.
   */
  protected function isFrontPagePath($path) {
    $front_page = \Drupal::config('system.site')->get('page.front');
    return $front_page && $path === $front_page;
  }

}
