<?php

declare(strict_types=1);

namespace Drupal\Tests\group_purl\Functional;

use Drupal\Core\Test\AssertMailTrait;
use Drupal\group\Entity\Group;
use Drupal\group\PermissionScopeInterface;
use Drupal\node\Entity\Node;
use Drupal\path_alias\Entity\PathAlias;
use Drupal\Tests\group\Functional\GroupBrowserTestBase;
use Drupal\user\RoleInterface;

/**
 * Functional tests for GroupPurlHooks entity insertion behavior.
 *
 * @group group_purl
 */
class GroupPurlHooksTest extends GroupBrowserTestBase {

  use AssertMailTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'path_alias',
    'group',
    'gnode',
    'purl',
    'group_purl',
  ];

  /**
   * A test user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * A test group.
   *
   * @var \Drupal\group\Entity\GroupInterface
   */
  protected $testGroup;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a node type.
    $this->drupalCreateContentType(['type' => 'page', 'name' => 'Basic page']);

    // Create a group type using the helper.
    $group_type = $this->createGroupType(['id' => 'default', 'label' => 'Default']);

    // Install group relationship type for nodes.
    $storage = $this->entityTypeManager->getStorage('group_relationship_type');
    $storage->save($storage->createFromPlugin($group_type, 'group_node:page'));

    // Create test user with permissions.
    $this->testUser = $this->drupalCreateUser([
      'access content',
      'bypass node access',
      'create page content',
    ]);

    // Create group role with permissions.
    $this->createGroupRole([
      'group_type' => 'default',
      'scope' => PermissionScopeInterface::INSIDER_ID,
      'global_role' => RoleInterface::AUTHENTICATED_ID,
      'permissions' => [
        'view group',
        'create group_node:page relationship',
        'view any group_node:page entity',
      ],
    ]);

    // Create test group.
    $this->testGroup = $this->createGroup([
      'type' => 'default',
      'label' => 'Test Group',
    ]);

    // Create path alias for group.
    PathAlias::create([
      'path' => '/group/' . $this->testGroup->id(),
      'alias' => '/testgroup',
      'langcode' => 'en',
    ])->save();

    // Add user as group member.
    $this->testGroup->addMember($this->testUser);

    // Set up PURL provider configuration.
    $this->config('purl.purl_provider.group_purl_provider')
      ->set('id', 'group_purl_provider')
      ->set('provider_key', 'group_purl_provider')
      ->set('label', 'Group')
      ->set('method_key', 'group_prefix')
      ->save();

    // Clear caches to ensure configuration is loaded.
    $this->rebuildAll();
  }

  /**
   * Test that creating a node via /[groupname]/node/add/page creates exactly one relationship.
   */
  public function testSingleRelationshipCreationViaGroupContext(): void {
    $this->drupalLogin($this->testUser);

    // Get initial relationship count.
    $initial_count = count($this->testGroup->getRelationships());

    // Create a new node programmatically while in group context.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node in Group Context',
      'status' => 1,
      'uid' => $this->testUser->id(),
    ]);

    // Mock the context provider to return our test group.
    $context_service = $this->createMock('\Drupal\group_purl\Context\GroupPurlContext');
    $mock_context = $this->createMock('\Drupal\Core\Plugin\Context\ContextInterface');
    $mock_context->method('getContextValue')->willReturn($this->testGroup);
    $context_service->method('getRuntimeContexts')->willReturn(['group' => $mock_context]);
    
    \Drupal::getContainer()->set('group_purl.context_provider', $context_service);

    // Mock route match to ensure we're not on entity_group_relationship route.
    $route_match = $this->createMock('\Drupal\Core\Routing\RouteMatchInterface');
    $route_match->method('getRouteName')->willReturn('entity.node.canonical');
    \Drupal::getContainer()->set('current_route_match', $route_match);
    
    // Save the node, which should trigger the entityInsert hook.
    $node->save();

    // Clear static cache to ensure we get fresh data.
    drupal_static_reset();
    \Drupal::entityTypeManager()->getStorage('group')->resetCache();
    \Drupal::entityTypeManager()->getStorage('group_relationship')->resetCache();

    // Get updated relationship count.
    $updated_count = count($this->testGroup->getRelationships());

    // Should have exactly one more relationship.
    $this->assertEquals($initial_count + 1, $updated_count, 'Exactly one relationship should be created when node is created in group context');

    // Verify the relationship exists and is of correct type.
    $relationships = $this->testGroup->getRelationshipsByEntity($node);
    $this->assertCount(1, $relationships, 'Node should have exactly one relationship with the group');
    
    $relationship = reset($relationships);
    $this->assertEquals('group_node:page', $relationship->getPlugin()->getPluginId(), 'Relationship should be of correct type');
  }

  /**
   * Test that multiple calls to entityInsert with same node/group don't create duplicates.
   */
  public function testStaticCachePreventsMultipleRelationships(): void {
    $this->drupalLogin($this->testUser);

    // Create a node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node for Static Cache',
      'status' => 1,
      'uid' => $this->testUser->id(),
    ]);
    $node->save();

    // Get initial relationship count.
    $initial_count = count($this->testGroup->getRelationships());

    // Simulate the entityInsert hook being called multiple times.
    $hooks = new \Drupal\group_purl\Hook\GroupPurlHooks();
    
    // Mock the context provider to return our test group.
    $context_service = $this->createMock('\Drupal\group_purl\Context\GroupPurlContext');
    $mock_context = $this->createMock('\Drupal\Core\Plugin\Context\ContextInterface');
    $mock_context->method('getContextValue')->willReturn($this->testGroup);
    $context_service->method('getRuntimeContexts')->willReturn(['group' => $mock_context]);
    
    \Drupal::getContainer()->set('group_purl.context_provider', $context_service);

    // Mock route match to ensure we're not on entity_group_relationship route.
    $route_match = $this->createMock('\Drupal\Core\Routing\RouteMatchInterface');
    $route_match->method('getRouteName')->willReturn('entity.node.canonical');
    \Drupal::getContainer()->set('current_route_match', $route_match);

    // Call entityInsert multiple times - should only create one relationship.
    $hooks->entityInsert($node);
    $hooks->entityInsert($node);
    $hooks->entityInsert($node);

    // Clear caches.
    drupal_static_reset();
    \Drupal::entityTypeManager()->getStorage('group')->resetCache();
    \Drupal::entityTypeManager()->getStorage('group_relationship')->resetCache();

    // Get updated relationship count.
    $updated_count = count($this->testGroup->getRelationships());

    // Should have exactly one more relationship despite multiple calls.
    $this->assertEquals($initial_count + 1, $updated_count, 'Static cache should prevent duplicate relationships from multiple hook calls');
    
    // Verify only one relationship exists for this node.
    $relationships = $this->testGroup->getRelationshipsByEntity($node);
    $this->assertCount(1, $relationships, 'Node should have exactly one relationship despite multiple hook calls');
  }

  /**
   * Test that entityInsert is skipped when on entity.group_relationship routes.
   */
  public function testSkipRelationshipCreationOnGroupRelationshipRoute(): void {
    $this->drupalLogin($this->testUser);

    // Create a node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node for Route Check',
      'status' => 1,
      'uid' => $this->testUser->id(),
    ]);
    $node->save();

    // Get initial relationship count.
    $initial_count = count($this->testGroup->getRelationships());

    // Mock the context provider to return our test group.
    $context_service = $this->createMock('\Drupal\group_purl\Context\GroupPurlContext');
    $mock_context = $this->createMock('\Drupal\Core\Plugin\Context\ContextInterface');
    $mock_context->method('getContextValue')->willReturn($this->testGroup);
    $context_service->method('getRuntimeContexts')->willReturn(['group' => $mock_context]);
    
    \Drupal::getContainer()->set('group_purl.context_provider', $context_service);

    // Mock route match to simulate being on entity.group_relationship.create_form route.
    $route_match = $this->createMock('\Drupal\Core\Routing\RouteMatchInterface');
    $route_match->method('getRouteName')->willReturn('entity.group_relationship.create_form');
    \Drupal::getContainer()->set('current_route_match', $route_match);

    // Call entityInsert - should be skipped due to route check.
    $hooks = new \Drupal\group_purl\Hook\GroupPurlHooks();
    $hooks->entityInsert($node);

    // Clear caches.
    drupal_static_reset();
    \Drupal::entityTypeManager()->getStorage('group')->resetCache();
    \Drupal::entityTypeManager()->getStorage('group_relationship')->resetCache();

    // Get updated relationship count.
    $updated_count = count($this->testGroup->getRelationships());

    // Should have the same number of relationships (no new ones created).
    $this->assertEquals($initial_count, $updated_count, 'No relationship should be created when on entity.group_relationship route');
    
    // Verify no relationship exists for this node.
    $relationships = $this->testGroup->getRelationshipsByEntity($node);
    $this->assertCount(0, $relationships, 'Node should have no relationships when created via group relationship form');
  }

  /**
   * Test route check with various entity.group_relationship route variations.
   */
  public function testRouteCheckWithVariousGroupRelationshipRoutes(): void {
    $this->drupalLogin($this->testUser);

    $test_routes = [
      'entity.group_relationship.create_form',
      'entity.group_relationship.edit_form',
      'entity.group_relationship.delete_form',
      'entity.group_relationship.canonical',
    ];

    foreach ($test_routes as $route_name) {
      // Create a new node for each test.
      $node = Node::create([
        'type' => 'page',
        'title' => 'Test Node for Route: ' . $route_name,
        'status' => 1,
        'uid' => $this->testUser->id(),
      ]);
      $node->save();

      // Get initial relationship count.
      $initial_count = count($this->testGroup->getRelationships());

      // Mock the context provider.
      $context_service = $this->createMock('\Drupal\group_purl\Context\GroupPurlContext');
      $mock_context = $this->createMock('\Drupal\Core\Plugin\Context\ContextInterface');
      $mock_context->method('getContextValue')->willReturn($this->testGroup);
      $context_service->method('getRuntimeContexts')->willReturn(['group' => $mock_context]);
      
      \Drupal::getContainer()->set('group_purl.context_provider', $context_service);

      // Mock route match for this specific route.
      $route_match = $this->createMock('\Drupal\Core\Routing\RouteMatchInterface');
      $route_match->method('getRouteName')->willReturn($route_name);
      \Drupal::getContainer()->set('current_route_match', $route_match);

      // Call entityInsert - should be skipped.
      $hooks = new \Drupal\group_purl\Hook\GroupPurlHooks();
      $hooks->entityInsert($node);

      // Clear caches.
      drupal_static_reset();
      \Drupal::entityTypeManager()->getStorage('group')->resetCache();
      \Drupal::entityTypeManager()->getStorage('group_relationship')->resetCache();

      // Get updated relationship count.
      $updated_count = count($this->testGroup->getRelationships());

      // Should have the same number of relationships for each route.
      $this->assertEquals($initial_count, $updated_count, "No relationship should be created when on route: {$route_name}");
    }
  }

  /**
   * Test that static cache is properly isolated per node/group combination.
   */
  public function testStaticCacheIsolationPerNodeGroupCombination(): void {
    $this->drupalLogin($this->testUser);

    // Create a second group.
    $second_group = $this->createGroup([
      'type' => 'default',
      'label' => 'Second Test Group',
    ]);
    $second_group->addMember($this->testUser);

    // Create two nodes.
    $node1 = Node::create([
      'type' => 'page',
      'title' => 'Test Node 1',
      'status' => 1,
      'uid' => $this->testUser->id(),
    ]);
    $node1->save();

    $node2 = Node::create([
      'type' => 'page',
      'title' => 'Test Node 2',
      'status' => 1,
      'uid' => $this->testUser->id(),
    ]);
    $node2->save();

    $hooks = new \Drupal\group_purl\Hook\GroupPurlHooks();

    // Mock route match to ensure we're not on entity_group_relationship route.
    $route_match = $this->createMock('\Drupal\Core\Routing\RouteMatchInterface');
    $route_match->method('getRouteName')->willReturn('entity.node.canonical');
    \Drupal::getContainer()->set('current_route_match', $route_match);

    // Test node1 with first group multiple times.
    $context_service1 = $this->createMock('\Drupal\group_purl\Context\GroupPurlContext');
    $mock_context1 = $this->createMock('\Drupal\Core\Plugin\Context\ContextInterface');
    $mock_context1->method('getContextValue')->willReturn($this->testGroup);
    $context_service1->method('getRuntimeContexts')->willReturn(['group' => $mock_context1]);
    
    \Drupal::getContainer()->set('group_purl.context_provider', $context_service1);

    $hooks->entityInsert($node1);
    $hooks->entityInsert($node1); // Should be cached

    // Test node1 with second group - should create new relationship.
    $context_service2 = $this->createMock('\Drupal\group_purl\Context\GroupPurlContext');
    $mock_context2 = $this->createMock('\Drupal\Core\Plugin\Context\ContextInterface');
    $mock_context2->method('getContextValue')->willReturn($second_group);
    $context_service2->method('getRuntimeContexts')->willReturn(['group' => $mock_context2]);
    
    \Drupal::getContainer()->set('group_purl.context_provider', $context_service2);

    $hooks->entityInsert($node1); // Different group, should create relationship

    // Test node2 with first group - should create new relationship.
    \Drupal::getContainer()->set('group_purl.context_provider', $context_service1);
    $hooks->entityInsert($node2);

    // Clear caches.
    drupal_static_reset();
    \Drupal::entityTypeManager()->getStorage('group')->resetCache();
    \Drupal::entityTypeManager()->getStorage('group_relationship')->resetCache();

    // Verify relationships.
    $node1_group1_relationships = $this->testGroup->getRelationshipsByEntity($node1);
    $this->assertCount(1, $node1_group1_relationships, 'Node1 should have exactly one relationship with first group');

    $node1_group2_relationships = $second_group->getRelationshipsByEntity($node1);
    $this->assertCount(1, $node1_group2_relationships, 'Node1 should have exactly one relationship with second group');

    $node2_group1_relationships = $this->testGroup->getRelationshipsByEntity($node2);
    $this->assertCount(1, $node2_group1_relationships, 'Node2 should have exactly one relationship with first group');
  }

  /**
   * Test that entities without group relationship plugins are not processed.
   */
  public function testEntitiesWithoutGroupPluginsNotAffected(): void {
    $this->drupalLogin($this->testUser);

    // Create a path alias entity - path aliases don't have group plugins.
    $alias = \Drupal::entityTypeManager()
      ->getStorage('path_alias')
      ->create([
        'path' => '/node/123',
        'alias' => '/test-alias',
        'langcode' => 'en',
      ]);
    $alias->save();

    // Get initial relationship count.
    $initial_count = count($this->testGroup->getRelationships());

    // Mock the context provider to return the test group.
    $context_service = $this->createMock('\Drupal\group_purl\Context\GroupPurlContext');
    $mock_context = $this->createMock('\Drupal\Core\Plugin\Context\ContextInterface');
    $mock_context->method('getContextValue')->willReturn($this->testGroup);
    $context_service->method('getRuntimeContexts')->willReturn(['group' => $mock_context]);
    
    \Drupal::getContainer()->set('group_purl.context_provider', $context_service);

    // Mock route match to ensure we're not on entity_group_relationship route.
    $route_match = $this->createMock('\Drupal\Core\Routing\RouteMatchInterface');
    $route_match->method('getRouteName')->willReturn('entity.path_alias.canonical');
    \Drupal::getContainer()->set('current_route_match', $route_match);

    // Call entityInsert with path alias - should be ignored because no plugin available.
    $hooks = new \Drupal\group_purl\Hook\GroupPurlHooks();
    $hooks->entityInsert($alias);

    // Clear caches.
    drupal_static_reset();
    \Drupal::entityTypeManager()->getStorage('group')->resetCache();
    \Drupal::entityTypeManager()->getStorage('group_relationship')->resetCache();

    // Get updated relationship count.
    $updated_count = count($this->testGroup->getRelationships());

    // Should have the same number of relationships (no new ones created).
    $this->assertEquals($initial_count, $updated_count, 'Path alias entities without available group relationship plugins should not trigger relationship creation');
  }

  /**
   * Test that entities with group relationship plugins ARE processed.
   */
  public function testEntitiesWithGroupPluginsAreProcessed(): void {
    // Install group_membership plugin if not already installed.
    if (!$this->testGroup->getGroupType()->hasPlugin('group_membership')) {
      $this->testGroup->getGroupType()->installPlugin('group_membership');
    }

    $this->drupalLogin($this->testUser);

    // Create a user entity.
    $test_user = $this->createUser(['access content']);

    // Get initial relationship count.
    $initial_count = count($this->testGroup->getRelationships());

    // Mock the context provider to return our test group.
    $context_service = $this->createMock('\Drupal\group_purl\Context\GroupPurlContext');
    $mock_context = $this->createMock('\Drupal\Core\Plugin\Context\ContextInterface');
    $mock_context->method('getContextValue')->willReturn($this->testGroup);
    $context_service->method('getRuntimeContexts')->willReturn(['group' => $mock_context]);
    
    \Drupal::getContainer()->set('group_purl.context_provider', $context_service);

    // Mock route match to ensure we're not on entity_group_relationship route.
    $route_match = $this->createMock('\Drupal\Core\Routing\RouteMatchInterface');
    $route_match->method('getRouteName')->willReturn('entity.node.canonical');
    \Drupal::getContainer()->set('current_route_match', $route_match);

    // Call entityInsert with user entity - should create membership.
    $hooks = new \Drupal\group_purl\Hook\GroupPurlHooks();
    $hooks->entityInsert($test_user);

    // Clear caches.
    drupal_static_reset();
    \Drupal::entityTypeManager()->getStorage('group')->resetCache();
    \Drupal::entityTypeManager()->getStorage('group_relationship')->resetCache();

    // Get updated relationship count.
    $updated_count = count($this->testGroup->getRelationships());

    // Should have exactly one more relationship.
    $this->assertEquals($initial_count + 1, $updated_count, 'Users should trigger group_membership relationship creation when plugin is available');

    // Verify the relationship is of correct type.
    $relationships = $this->testGroup->getRelationshipsByEntity($test_user);
    $this->assertCount(1, $relationships, 'User should have exactly one relationship with the group');
    
    $relationship = reset($relationships);
    $this->assertEquals('group_membership', $relationship->getPlugin()->getPluginId(), 'Relationship should be group_membership type');
  }

}