<?php

declare(strict_types=1);

namespace Drupal\Tests\group_purl\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\group_purl\Plugin\Purl\Method\GroupPrefixMethod;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Psr\Log\LoggerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Unit tests for GroupPrefixMethod.
 *
 * @coversDefaultClass \Drupal\group_purl\Plugin\Purl\Method\GroupPrefixMethod
 * @group group_purl
 */
class GroupPrefixMethodTest extends UnitTestCase {

  /**
   * The group prefix method under test.
   *
   * @var \Drupal\group_purl\Plugin\Purl\Method\GroupPrefixMethod
   */
  protected GroupPrefixMethod $method;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set up a mock container with logger services.
    $container = new ContainerBuilder();
    $logger = $this->createMock(LoggerInterface::class);
    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $loggerFactory->method('get')->with('group_purl')->willReturn($logger);

    $container->set('logger.factory', $loggerFactory);
    $container->set('logger.channel.group_purl', $logger);
    \Drupal::setContainer($container);

    $this->method = new GroupPrefixMethod([], 'group_prefix', []);
  }

  /**
   * Tests contains method with exact modifier match.
   *
   * @covers ::contains
   */
  public function testContainsExactMatch(): void {
    $request = Request::create('/council');
    $result = $this->method->contains($request, 'council');
    $this->assertTrue($result);
  }

  /**
   * Tests contains method with prefixed path.
   *
   * @covers ::contains
   */
  public function testContainsPrefixedPath(): void {
    $request = Request::create('/council/test-page');
    $result = $this->method->contains($request, 'council');
    $this->assertTrue($result);
  }

  /**
   * Tests contains method with non-matching path.
   *
   * @covers ::contains
   */
  public function testContainsNoMatch(): void {
    $request = Request::create('/other-path');
    $result = $this->method->contains($request, 'council');
    $this->assertFalse($result);
  }

  /**
   * Tests contains method with partial match.
   *
   * @covers ::contains
   */
  public function testContainsPartialMatch(): void {
    $request = Request::create('/councils');
    $result = $this->method->contains($request, 'council');
    $this->assertFalse($result);
  }

  /**
   * Tests alterRequest method with prefixed path.
   *
   * @covers ::alterRequest
   */
  public function testAlterRequestWithPrefixedPath(): void {
    $request = Request::create('/council/test-page?param=value');
    $request->server->set('REQUEST_URI', '/council/test-page?param=value');

    $result = $this->method->alterRequest($request, 'council');

    $this->assertInstanceOf(Request::class, $result);
    $this->assertEquals('/test-page?param=value', $result->server->get('REQUEST_URI'));
  }

  /**
   * Tests alterRequest method with root path.
   *
   * @covers ::alterRequest
   */
  public function testAlterRequestWithRootPath(): void {
    $request = Request::create('/council/');
    $request->server->set('REQUEST_URI', '/council/');

    $result = $this->method->alterRequest($request, 'council');

    // Should return TrustedRedirectResponse to redirect to URL without trailing slash
    $this->assertInstanceOf(TrustedRedirectResponse::class, $result);
    $this->assertEquals('/council', $result->getTargetUrl());
  }

  /**
   * Tests alterRequest method with exact modifier match.
   *
   * @covers ::alterRequest
   */
  public function testAlterRequestExactMatch(): void {
    $request = Request::create('/council');
    $request->server->set('REQUEST_URI', '/council');

    $result = $this->method->alterRequest($request, 'council');

    $this->assertFalse($result);
  }

  /**
   * Tests alterRequest method with duplicate modifier.
   *
   * @covers ::alterRequest
   */
  public function testAlterRequestDuplicateModifier(): void {
    $request = Request::create('/council/council/test');
    $request->server->set('REQUEST_URI', '/council/council/test');

    $result = $this->method->alterRequest($request, 'council');

    $this->assertInstanceOf(TrustedRedirectResponse::class, $result);
    $this->assertEquals('/council/test', $result->getTargetUrl());
  }

  /**
   * Tests enterContext method.
   *
   * @covers ::enterContext
   */
  public function testEnterContext(): void {
    $path = '/test-page';
    $options = [];

    $result = $this->method->enterContext('council', $path, $options);

    $this->assertEquals('/council/test-page', $result);
  }

  /**
   * Tests enterContext method with exit option.
   *
   * @covers ::enterContext
   */
  public function testEnterContextWithExit(): void {
    $path = '/test-page';
    $options = ['purl_exit' => TRUE];

    $result = $this->method->enterContext('council', $path, $options);

    $this->assertEquals('/test-page', $result);
  }

  /**
   * Tests exitContext method with prefixed path.
   *
   * @covers ::exitContext
   */
  public function testExitContextWithPrefixedPath(): void {
    $path = '/council/test-page';
    $options = [];

    $result = $this->method->exitContext('council', $path, $options);

    $this->assertEquals('/council', $result);
  }

  /**
   * Tests exitContext method with non-prefixed path.
   *
   * @covers ::exitContext
   */
  public function testExitContextWithNonPrefixedPath(): void {
    $path = '/other-page';
    $options = [];

    $result = $this->method->exitContext('council', $path, $options);

    $this->assertNull($result);
  }

}
